package ru.yandex.autotests.direct.api.ads.update.textad;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.AdGetItem;
import com.yandex.direct.api.v5.ads.GetResponse;
import com.yandex.direct.api.v5.ads.TextAdFieldEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.api.ads.AdsStories;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.ads.AdGetItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdGetMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.RandomStringUtils.randomAlphabetic;
import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanEquivalentV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by semkagtn on 13.04.15.
 * https://st.yandex-team.ru/TESTIRT-5113
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Stories(AdsStories.TEXT_AD)
@Description("Проверка сохранения ссылки при обновлении текстового объявления")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class UpdateHrefTest {

    private static final String LOGIN = AdsLogins.CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static long adID;

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public String href;

    @Parameterized.Parameter(2)
    public String expectedHref;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        String httpHref = "http://yandex.ru";
        String httpsHref = "https://ya.ru";
        String minDomainLengthHref = "http://" + randomAlphabetic(TextAdAddMap.MIN_DOMAIN_LENGTH) + ".ru";
        String maxDomainLengthHref = "http://" + randomAlphabetic(TextAdAddMap.MAX_DOMAIN_LENGTH) +
                "." + randomAlphabetic(TextAdAddMap.MAX_DOMAIN_LENGTH) + ".ru";
        String minTopDomainLengthHref = "http://test." + randomAlphabetic(TextAdAddMap.MIN_TOP_DOMAIN_LENGTH);
        String maxTopDomainLengthHref = "http://tester." + randomAlphabetic(TextAdAddMap.MAX_TOP_DOMAIN_LENGTH);
        String russianHref = "http://подарки.рф";
        String ukrainianHref = "http://сувеніри.укр";
        String numericHref = "http://12345.b45";
        String dashedHref = "http://hello-world.ru";
        String maxLengthHref = "http://" + generateValidHrefWithoutProtocolOfLength(TextAdAddMap.MAX_HREF_LENGTH - 7);

        return Arrays.asList(new Object[][]{
                {"Обновить объявление со ссылкой, протокол которой http", httpHref, httpHref},
                {"Обновить объявление со ссылкой, протокол которой https", httpsHref, httpsHref},
                {"Обновить объявление со ссылкой с внутренним доменом длиной " + TextAdAddMap.MIN_DOMAIN_LENGTH,
                        minDomainLengthHref, minDomainLengthHref},
                {"Обновить объявление со ссылкой с внутренними доменами длиной " + TextAdAddMap.MAX_DOMAIN_LENGTH,
                        maxDomainLengthHref, maxDomainLengthHref},
                {"Обновить объявление со ссылкой с доменом верхнего уровня длиной " + TextAdAddMap.MIN_TOP_DOMAIN_LENGTH,
                        minTopDomainLengthHref, minTopDomainLengthHref},
                {"Обновить объявление со ссылкой с доменом верхнего уровня длиной " + TextAdAddMap.MAX_TOP_DOMAIN_LENGTH,
                        maxTopDomainLengthHref, maxTopDomainLengthHref},
                {"Обновить объявление со ссылкой, состоящей из русских букв", russianHref, russianHref},
                {"Обновить объявление со ссылкой, состоящей из украинских букв", ukrainianHref, ukrainianHref},
                {"Обновить объявление со ссылкой, содержащей цифры", numericHref, numericHref},
                {"Обновить объявление со ссылкой, содержащей дефисы", dashedHref, dashedHref},
                {"Обновить объявление со ссылкой, длиной " + TextAdAddMap.MAX_HREF_LENGTH, maxLengthHref, maxLengthHref}
        });
    }

    @BeforeClass
    public static void createAd() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        adID = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
    }

    @Before
    public void addUnitsToClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    public void updateHref() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(adID)
                                .withTextAd(new TextAdUpdateMap()
                                        .defaultAd()
                                        .withHref(href))),
                ExpectedResult.success(adID));

        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.HREF)
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(adID)));
        assertThat("ссылка сохранилась верно", response.getAds(), beanEquivalentV5(Arrays.asList(
                (AdGetItem) new AdGetItemMap()
                        .withTextAd(new TextAdGetMap()
                                .withHref(href)).getBean())));
    }


    private static String generateValidHrefWithoutProtocolOfLength(int length) {
        StringBuilder stringBuilder = new StringBuilder(randomAlphabetic(length));
        int domainLength = Math.min(
                (TextAdAddMap.MAX_DOMAIN_LENGTH - 1) / 2, (TextAdAddMap.MAX_TOP_DOMAIN_LENGTH - 1) / 2);
        for (int i = domainLength; i < length - domainLength; i += domainLength) {
            stringBuilder.setCharAt(i, '.');
        }
        return stringBuilder.toString();
    }
}
