package ru.yandex.autotests.direct.api.ads.update.textad;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersLanguage;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.PhrasesRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Arrays.asList;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static ru.yandex.autotests.directapi.model.common.RegionIDValues.BELARUS_REGION_ID;
import static ru.yandex.autotests.directapi.model.common.RegionIDValues.GERMANY_REGION_ID;
import static ru.yandex.autotests.directapi.model.common.RegionIDValues.KAZAKH_REGION_ID;
import static ru.yandex.autotests.directapi.model.common.RegionIDValues.RUSSIA_REGION_ID;
import static ru.yandex.autotests.directapi.model.common.RegionIDValues.TURKEY_REGION_ID;
import static ru.yandex.autotests.directapi.model.common.RegionIDValues.UKRAINIAN_REGION_ID;

@Aqua.Test
@Description("Определение и обновление в базе языка баннера")
@Issue("https://st.yandex-team.ru/DIRECT-102426")
@Features(AdsFeatures.UPDATE)
@RunWith(Parameterized.class)
public class UpdateTextAdLanguageTest {

    private static final String CLIENT = AdsLogins.CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public String bannerText;

    @Parameterized.Parameter(1)
    public BannersLanguage expectedLanguage;

    @Parameterized.Parameter(2)
    public String regionId;

    @Parameterized.Parameters(name = "bannerText = {0}, expectedLanguage = {1}, regionId = {2}")
    public static Iterable<Object[]> data() {
        return asList(
                new Object[] {"Продам гараж", BannersLanguage.ru_, RUSSIA_REGION_ID},
                new Object[] {"Sell fridge", BannersLanguage.en, RUSSIA_REGION_ID},
                new Object[] {"Таварыства кнігалюбаў", BannersLanguage.be, BELARUS_REGION_ID},
                new Object[] {"Haus Des Volkes Das Bauhaushotel", BannersLanguage.de, GERMANY_REGION_ID},
                new Object[] {"Балалар киімін немен жуады", BannersLanguage.kk, KAZAKH_REGION_ID},
                new Object[] {"45 Numara Ayakkabı Modelleri", BannersLanguage.tr, TURKEY_REGION_ID},
                new Object[] {"Офіційний сайт турецьких авіаліній", BannersLanguage.uk, UKRAINIAN_REGION_ID}
        );
    }

    private static DirectJooqDbSteps jooqDbSteps;

    private Long adGroupId;
    private Long bannerId;

    @BeforeClass
    public static void beforeClass() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT);
    }

    @Before
    public void before() {
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(AdsLogins.CLIENT);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        bannerId = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
    }

    @Test
    public void languageTest() {
        updateGeo();

        api.userSteps.adsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withTextAd(new TextAdUpdateMap().withTitle(bannerText).withText(bannerText))
                                .withId(bannerId)),
                ExpectedResult.success());

        BannersRecord bannersRecord = jooqDbSteps.bannersSteps().getBanner(bannerId);
        BannersLanguage actualLanguage = bannersRecord.getLanguage();

        assertThat(actualLanguage, is(expectedLanguage));
    }

    private void updateGeo() {
        PhrasesRecord phrasesRecord = jooqDbSteps.adGroupsSteps().getPhrases(adGroupId);
        phrasesRecord.setGeo(regionId);
        jooqDbSteps.adGroupsSteps().updatePhrases(phrasesRecord);
    }
}

