package ru.yandex.autotests.direct.api.ads.update.textad;

import java.util.Arrays;
import java.util.Collection;

import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;

/**
 * Created by pavryabov on 29.06.17.
 * https://st.yandex-team.ru/DIRECT-67387
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Description("Проверки обновления статусов после различных изменений заголовков")
@Issue("https://st.yandex-team.ru/DIRECT-65773")
@RunWith(Parameterized.class)
public class UpdateTitlesCheckStatusesTest {

    private static final String LOGIN = AdsLogins.CLIENT;

    public static final String OLD_TITLE = "подарки,новогодние";
    public static final String OLD_TITLE_WITH_MINOR_CHANGE = "подарки, новогодние";
    public static final String NEW_TITLE = "елки";
    public static final String OLD_TITLE_2 = "коты,собаки";
    public static final String OLD_TITLE_2_WITH_MINOR_CHANGE = "коты, собаки";
    public static final String NEW_TITLE_2 = "панды";

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long campaignId;
    private static Long adGroupId;
    private static Long vCardId;
    private Long adId;
    private static DateTime lastChange = DateTime.now().minusDays(1);
    private static final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");

    @Parameterized.Parameter(0)
    public String titleBefore;

    @Parameterized.Parameter(1)
    public String title2Before;

    @Parameterized.Parameter(2)
    public String titleAfter;

    @Parameterized.Parameter(3)
    public String title2After;

    @Parameterized.Parameter(4)
    public String expectedStatusBsSynced;

    @Parameterized.Parameter(5)
    public String expectedStatusModerate;

    @Parameterized.Parameters(name = "Title: {0} -> {2}, Title2: {1} -> {3}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {OLD_TITLE, OLD_TITLE_2, NEW_TITLE, NEW_TITLE_2, Status.NO, Status.READY},
                {OLD_TITLE, OLD_TITLE_2, NEW_TITLE, OLD_TITLE_2, Status.NO, Status.READY},
                {OLD_TITLE, OLD_TITLE_2, NEW_TITLE, OLD_TITLE_2_WITH_MINOR_CHANGE, Status.NO, Status.READY},
                {OLD_TITLE, OLD_TITLE_2, OLD_TITLE_WITH_MINOR_CHANGE, NEW_TITLE_2, Status.NO, Status.READY},
                {OLD_TITLE, OLD_TITLE_2, OLD_TITLE_WITH_MINOR_CHANGE, OLD_TITLE_2, Status.NO, Status.YES},
                {OLD_TITLE, OLD_TITLE_2, OLD_TITLE_WITH_MINOR_CHANGE, OLD_TITLE_2_WITH_MINOR_CHANGE, Status.NO, Status.YES},
                {OLD_TITLE, OLD_TITLE_2, OLD_TITLE, NEW_TITLE_2, Status.NO, Status.READY},
                {OLD_TITLE, OLD_TITLE_2, OLD_TITLE, OLD_TITLE_2, Status.YES, Status.YES},
                {OLD_TITLE, OLD_TITLE_2, OLD_TITLE, OLD_TITLE_2_WITH_MINOR_CHANGE, Status.NO, Status.YES},
                {OLD_TITLE, null, OLD_TITLE, NEW_TITLE_2, Status.NO, Status.READY},
                {OLD_TITLE, OLD_TITLE_2, OLD_TITLE, null, Status.NO, Status.READY},
                {OLD_TITLE, null, OLD_TITLE, null, Status.YES, Status.YES},
                {OLD_TITLE, null, OLD_TITLE_WITH_MINOR_CHANGE, null, Status.NO, Status.YES},
                {OLD_TITLE, null, NEW_TITLE, null, Status.NO, Status.READY},
        });
    }

    @BeforeClass
    public static void prepareData() {

        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN).campaignsSteps()
                .setCampaignsStatusModerate(campaignId, CampaignsStatusmoderate.Yes);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        vCardId = api.userSteps.vCardsSteps().addDefaultVCard(campaignId);
        api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void setAdStatuses() {
        adId = api.userSteps.adsSteps().addAd(new AdAddItemMap()
                .withAdGroupId(adGroupId)
                .withTextAd(new TextAdAddMap()
                        .defaultTextAd()
                        .withTitle(titleBefore)
                        .withTitle2(title2Before)
                        .withVCardId(vCardId)));
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(adId);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adId, Status.YES);
        api.userSteps.bannersFakeSteps().setLastChange(adId, lastChange.toString());

        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    public void updateTitles() {
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adId)
                        .withTextAd(new TextAdUpdateMap()
                                .withTitle(titleAfter)
                                .withTitle2(title2After))));

        BannerFakeInfo actualBannerFakeInfo = api.userSteps.bannersFakeSteps().getBannerParams(adId);
        BannerFakeInfo expectedBannerFakeInfo = new BannerFakeInfo();
        expectedBannerFakeInfo.setStatusBsSynced(expectedStatusBsSynced);
        expectedBannerFakeInfo.setStatusModerate(expectedStatusModerate);
        expectedBannerFakeInfo.setPhoneFlag(expectedStatusModerate);

        assertThat("статусы объявления обновлись верно",
                actualBannerFakeInfo,
                beanDiffer(expectedBannerFakeInfo).useCompareStrategy(DefaultCompareStrategies.onlyExpectedFields()));

        LocalDate actualDate = formatter.parseLocalDate(actualBannerFakeInfo.getLastChange());
        assertThat("LastChange объявления обновился", actualDate, not(equalTo(lastChange)));
    }
}
