package ru.yandex.autotests.direct.api.ads.update.textad;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.AdGetItem;
import com.yandex.direct.api.v5.ads.GetResponse;
import com.yandex.direct.api.v5.ads.TextAdFieldEnum;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.api.ads.AdsStories;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdGetItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdGetMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanEquivalentV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by semkagtn on 09.04.15.
 * https://st.yandex-team.ru/TESTIRT-5113
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Stories(AdsStories.TEXT_AD)
@Description("Проверка обновления визитной карточки объявлений")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class UpdateVCardTest {

    private static final String LOGIN = AdsLogins.CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static long adGroupID;

    private static DateTime lastChange = DateTime.now().minusDays(3);
    private static final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public Long oldVCardID;

    @Parameterized.Parameter(2)
    public Long newVCardID;

    @Parameterized.Parameter(3)
    public ProtocolType protocol;

    @Parameterized.Parameters(name = "{0}, protocol = {3}")
    public static Collection<Object[]> data() {
        api.protocol(ProtocolType.JSON);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        adGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);
        long firstVCardID = api.userSteps.vCardsSteps().addDefaultVCard(campaignID);
        long secondVCardID = api.userSteps.vCardsSteps().addDefaultVCard(campaignID);

        return Arrays.asList(new Object[][]{
                {"Привязка визитной карточки к объявлению", null, firstVCardID, ProtocolType.JSON},
                {"Изменение визитной карточки в объявлении", firstVCardID, secondVCardID, ProtocolType.JSON},
                {"Отвязка визитной карточки от объявления", firstVCardID, null, ProtocolType.JSON},
                {"Отвязка визитной карточки от объявления", firstVCardID, null, ProtocolType.SOAP},
        });
    }

    private long adID;

    @Before
    @Step("Подготовка данных для теста")
    public void createAd() {
        api.protocol(ProtocolType.JSON);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        adID = api.userSteps.adsSteps().addAd(new AdAddItemMap()
                .withAdGroupId(adGroupID)
                .withTextAd(new TextAdAddMap()
                        .defaultTextAd()
                        .withVCardId(oldVCardID)));
    }

    @Test
    public void updateVCard() {
        api.protocol(protocol);
        api.userSteps.adsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(adID)
                                .withTextAd(new TextAdUpdateMap()
                                        .withVCardId(newVCardID))),
                ExpectedResult.success(adID));

        api.protocol(ProtocolType.JSON);
        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.V_CARD_ID)
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(adID)));
        assertThat("визитная карточка в объявлении была успешно изменена", response.getAds(),
                beanEquivalentV5(Arrays.asList((AdGetItem) new AdGetItemMap()
                        .withTextAd(new TextAdGetMap()
                                .withVCardId(newVCardID)).getBean())));
    }

    @Test
    public void checkStatusesAfterUpdate() {
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(adID);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adID, Status.YES);
        api.userSteps.bannersFakeSteps().setLastChange(adID, lastChange.toString());
        api.protocol(protocol);
        api.userSteps.adsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(adID)
                                .withTextAd(new TextAdUpdateMap()
                                        .withVCardId(newVCardID))),
                ExpectedResult.success(adID));

        BannerFakeInfo actualBannerFakeInfo = api.userSteps.bannersFakeSteps().getBannerParams(adID);

        BannerFakeInfo expectedBannerFakeInfo = new BannerFakeInfo();
        expectedBannerFakeInfo.setStatusBsSynced(Status.NO);
        expectedBannerFakeInfo.setStatusModerate(Status.YES);

        assertThat("статусы объявления обновлись верно",
                actualBannerFakeInfo, beanEquivalent(expectedBannerFakeInfo));

        LocalDate actualDate = formatter.parseLocalDate(actualBannerFakeInfo.getLastChange());
        assertThat("LastChange объявления обновился", actualDate, equalTo(LocalDate.now()));
    }
}
