package ru.yandex.autotests.direct.api.ads.update.textad.calloutsetting;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.general.OperationEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.adextensiontypes.AdExtensionSettingItemMap;
import ru.yandex.autotests.directapi.model.api5.adextensiontypes.AdExtensionSettingMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.RandomStringUtils.randomAlphabetic;
import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.junit.runners.Parameterized.Parameter;
import static org.junit.runners.Parameterized.Parameters;
import static ru.yandex.autotests.directapi.model.api5.general.ExpectedResult.errors;
import static ru.yandex.autotests.directapi.model.api5.general.ExpectedResult.warnings;
import static ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult.Common.fieldMustBePositiveInteger;
import static ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult.bothJavaAndPerl;

/**
 * Created by pavryabov on 15.03.16.
 * https://st.yandex-team.ru/TESTIRT-8642
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Description("Неверные идентификаторы уточнений в запросе на обновление текстового объявления")
@Issue("https://st.yandex-team.ru/DIRECT-50726")
@RunWith(Parameterized.class)
public class UpdateInvalidAdExtensionIdTest {

    private static final String LOGIN = AdsLogins.CLIENT;
    private static final String ANOTHER_LOGIN = AdsLogins.CLIENT_ELSE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long adId;

    @Parameter(0)
    public String description;

    @Parameter(1)
    public Long invalidAdExtensionId;

    @Parameter(2)
    public JavaOrPerlExpectedResult expectedAddAndSetResult;

    @Parameter(3)
    public JavaOrPerlExpectedResult expectedRemoveResult;

    @Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(ANOTHER_LOGIN);

        api.as(ANOTHER_LOGIN);
        Long anotherAdExtensionId = api.userSteps.adExtensionsSteps().addCalloutWithText(randomAlphabetic(15));

        api.as(LOGIN);
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        adId = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        Long adExtensionId = api.userSteps.adExtensionsSteps().addCalloutWithText(randomAlphabetic(15));
        Long nonexistentAdExtensionId = adExtensionId + 300000;
        return Arrays.asList(new Object[][]{
                {"несуществующий идентификатор уточнения", nonexistentAdExtensionId,
                        bothJavaAndPerl(8800, Api5ErrorDetails.AD_EXTENSION_NOT_FOUND, nonexistentAdExtensionId),
                        notLinkedError(adId, nonexistentAdExtensionId)},
                {"чужой идентификатор уточнения", anotherAdExtensionId,
                        bothJavaAndPerl(8800, Api5ErrorDetails.AD_EXTENSION_NOT_FOUND, anotherAdExtensionId),
                        notLinkedError(adId, anotherAdExtensionId)},
                {"нулевой идентификатор уточнения", 0l,
                        fieldMustBePositiveInteger(capitalize(TextAdUpdateMap.CALLOUT_SETTING) + "." +
                                capitalize(AdExtensionSettingMap.AD_EXTENSIONS) + "[1]." +
                                capitalize(AdExtensionSettingItemMap.AD_EXTENSION_ID)),
                        fieldMustBePositiveInteger(capitalize(TextAdUpdateMap.CALLOUT_SETTING) + "." +
                                capitalize(AdExtensionSettingMap.AD_EXTENSIONS) + "[1]." +
                                capitalize(AdExtensionSettingItemMap.AD_EXTENSION_ID))},
                {"отрицательный идентификатор уточнения", -1l,
                        fieldMustBePositiveInteger(capitalize(TextAdUpdateMap.CALLOUT_SETTING) + "." +
                                capitalize(AdExtensionSettingMap.AD_EXTENSIONS) + "[1]." +
                                capitalize(AdExtensionSettingItemMap.AD_EXTENSION_ID)),
                        fieldMustBePositiveInteger(capitalize(TextAdUpdateMap.CALLOUT_SETTING) + "." +
                                capitalize(AdExtensionSettingMap.AD_EXTENSIONS) + "[1]." +
                                capitalize(AdExtensionSettingItemMap.AD_EXTENSION_ID))},
        });
    }

    @Before
    public void addUnitsToUser() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    public void addInvalidAdExtensionId() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(adId)
                                .withTextAd(new TextAdUpdateMap()
                                        .defaultAd()
                                        .withCalloutSetting(new AdExtensionSettingMap()
                                                .withAdExtensions(new AdExtensionSettingItemMap()
                                                        .withAdExtensionId(invalidAdExtensionId)
                                                        .withOperation(OperationEnum.ADD))))),
                expectedAddAndSetResult);
    }

    @Test
    public void removeInvalidAdExtensionId() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(adId)
                                .withTextAd(new TextAdUpdateMap()
                                        .defaultAd()
                                        .withCalloutSetting(new AdExtensionSettingMap()
                                                .withAdExtensions(new AdExtensionSettingItemMap()
                                                        .withAdExtensionId(invalidAdExtensionId)
                                                        .withOperation(OperationEnum.REMOVE))))),
                expectedRemoveResult);
    }

    @Test
    public void setInvalidAdExtensionId() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(adId)
                                .withTextAd(new TextAdUpdateMap()
                                        .defaultAd()
                                        .withCalloutSetting(new AdExtensionSettingMap()
                                                .withAdExtensions(new AdExtensionSettingItemMap()
                                                        .withAdExtensionId(invalidAdExtensionId)
                                                        .withOperation(OperationEnum.SET))))),
                expectedAddAndSetResult);
    }

    private static JavaOrPerlExpectedResult notLinkedError(long adId, long adExtensionId) {
        return JavaOrPerlExpectedResult
                .java(warnings(adId, new Notification(10171, Api5ErrorDetails.AD_EXTENSION_NOT_LINKED, adExtensionId)))
                .perl(errors(new Notification(8800, Api5ErrorDetails.AD_EXTENSION_NOT_FOUND, adExtensionId)));
    }
}
