package ru.yandex.autotests.direct.api.ads.update.textimagead;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannerImagesFormatsImageType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.ImagesStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.ImagesRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.Image;
import ru.yandex.autotests.directapi.model.ValidImageData;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextImageAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by gerdler on 12.07.16.
 * https://st.yandex-team.ru/TESTIRT-9322
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Description("Обновление AdImageHash у графического текстового объявления")
@Issue("https://st.yandex-team.ru/DIRECT-54754")
public class UpdateTextImageAdAdImageHashTest {
    private static String client = AdsLogins.CLIENT46;
    private static String otherClient = AdsLogins.CLIENT47;

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long campaignId;
    private static Long adGroupId;
    private static DirectJooqDbSteps dbSteps;
    private static String smallAdImageHash;

    private Long adId;

    @BeforeClass
    public static void initTest() {
        api.as(client).userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        dbSteps = api.userSteps.getDarkSideSteps().getDirectJooqDbSteps().useShardForLogin(client);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
    }

    @Before
    public void setUp() {
        String adImageHash = api.userSteps.imagesStepsV5().addImageAdAdImage(2);
        adId = api.userSteps.adsSteps().addDefaultTextImageAd(adGroupId, adImageHash);

        dbSteps.bannersSteps().setBannerStatusModerate(adId, BannersStatusmoderate.Yes);
        dbSteps.imagesSteps().setImageStatusModerate(adId, adImageHash, ImagesStatusmoderate.Yes);
    }

    @Test
    public void updateWithAdImageHashSameSize() {
        String adImageHash = api.userSteps.imagesStepsV5().addImageAdAdImage(3);
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withAdImageHash(adImageHash)
                        ).withId(adId)
                ),
                ExpectedResult.success()
        );

        ImagesRecord image = dbSteps.imagesSteps().getImagesRecords(campaignId, adGroupId, adId);
        assertThat("к объявлению в БД привязано правильное изображение",
                image.getImageHash(), equalTo(adImageHash));
        assertThat("у изображения сбосился статус модерации",
                image.getStatusmoderate(), equalTo(ImagesStatusmoderate.Ready));
    }

    @Test
    public void updateWithAdImageHashDifferentSize() {
        String adImageHash = api.userSteps.imagesStepsV5().addImageAdAdImage(1);
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withAdImageHash(adImageHash)
                        ).withId(adId)
                ),
                ExpectedResult.errors(new Notification(6000, Api5ErrorDetails.DIFFERENT_AD_IMAGE_SIZE_FOR_THIS_BANNER_IN_UPDATE))
        );
    }

    @Test
    public void updateWithNotOwnAdImageHashSameSize() {
        String notOwnAdImageHash = api.as(otherClient).userSteps.imagesStepsV5().addImageAdAdImage(5);
        api.as(client).userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withAdImageHash(notOwnAdImageHash)
                        ).withId(adId)
                ),
                ExpectedResult.errors(
                        new Notification(8800, Api5ErrorDetails.IMAGE_NOT_FOUND)
                )
        );
    }

    @Test
    public void updateWithNotOwnAdImageHashDifferentSize() {
        String notOwnAdImageHash = api.as(otherClient).userSteps.imagesStepsV5().addImageAdAdImage(4);
        api.as(client).userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withAdImageHash(notOwnAdImageHash)
                        ).withId(adId)
                ),
                ExpectedResult.errors(
                        new Notification(8800, Api5ErrorDetails.IMAGE_NOT_FOUND)
                )
        );
    }

    @Test
    public void updateWithRegularAdImageHash() {
        String regularAdImageHash = api.userSteps.imagesStepsV5().addDefaultRegularAdImage();
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withAdImageHash(regularAdImageHash)
                        ).withId(adId)
                ),
                ExpectedResult.errors(
                        new Notification(6000, Api5ErrorDetails.INVALID_AD_IMAGE_TYPE_FOR_THIS_BANNER)
                )
        );
    }

    @Test
    public void updateWithWideAdImageHash() {
        String wideAdImageHash = api.userSteps.imagesStepsV5().addDefaultWideAdImage();
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withAdImageHash(wideAdImageHash)
                        ).withId(adId)
                ),
                ExpectedResult.errors(
                        new Notification(6000, Api5ErrorDetails.INVALID_AD_IMAGE_TYPE_FOR_THIS_BANNER)
                )
        );
    }

    @Test
    public void updateWithSmallAdImageHash() {
        Image image = ValidImageData.getFakeSmallImage(1);
        smallAdImageHash = api.userSteps.imagesStepsV5().addImage(image);
        dbSteps.imagesSteps().setImageType(smallAdImageHash, BannerImagesFormatsImageType.small);
        assumeThat("установился правильный тип изображения",
                dbSteps.imagesSteps().getBannerImagesFormatsRecords(smallAdImageHash).getImageType(),
                equalTo(BannerImagesFormatsImageType.small));
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withAdImageHash(smallAdImageHash)
                        ).withId(adId)
                ),
                ExpectedResult.errors(
                        new Notification(6000, Api5ErrorDetails.INVALID_AD_IMAGE_TYPE_FOR_THIS_BANNER)
                )
        );
    }

    @Test
    public void updateWithNonExistentAdImageHash() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withAdImageHash("11111111")
                        ).withId(adId)
                ),
                ExpectedResult.errors(
                        new Notification(8800, Api5ErrorDetails.IMAGE_NOT_FOUND)
                )
        );
    }

    @Test
    public void updateWithDeletedAdImageHash() {
        String adImageHash = api.userSteps.imagesStepsV5().addImageAdAdImage(4);
        api.userSteps.imagesStepsV5().adImagesDelete(adImageHash);

        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withAdImageHash(adImageHash)
                        ).withId(adId)
                ),
                ExpectedResult.errors(
                        new Notification(8800, Api5ErrorDetails.IMAGE_NOT_FOUND)
                )
        );
    }

    @Test
    public void updateWithEmptyAdImageHash() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withAdImageHash("")
                        ).withId(adId)
                ),
                ExpectedResult.errors(
                        new Notification(8800, Api5ErrorDetails.IMAGE_NOT_FOUND)
                )
        );
    }

    @Test
    public void updateWithSpaceInAdImageHash() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withAdImageHash(" ")
                        ).withId(adId)
                ),
                ExpectedResult.errors(
                        new Notification(8800, Api5ErrorDetails.IMAGE_NOT_FOUND)
                )
        );
    }

    @AfterClass
    public static void removeSmallImage() {
        dbSteps.imagesSteps().setImageType(smallAdImageHash, BannerImagesFormatsImageType.image_ad);
    }
}
