package ru.yandex.autotests.direct.api.ads.update.textimagead;

import java.time.LocalDateTime;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusbssynced;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.ImagesStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.ImagesRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextImageAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.anyOf;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;

/**
 * Created by gerdler on 13.07.16.
 * https://st.yandex-team.ru/TESTIRT-9322
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Description("Проверка обновления статусов (StatusBsSynced и StatusModerate) и LastChange " +
        "при обновлении текстового графического объявления")
@Issue("https://st.yandex-team.ru/DIRECT-54754")
@Tag(TagDictionary.TRUNK)
public class UpdateTextImageAdStatusesTest {

    private static final String CLIENT = AdsLogins.CLIENT48;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long adId;
    private static Long adGroupId;
    private static Long campaignId;
    private static LocalDateTime previousLastChange = LocalDateTime.now().withNano(0).minusDays(1);
    private static DirectJooqDbSteps dbSteps;

    private String adImageHash;

    @BeforeClass
    public static void prepareData() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);

        dbSteps = api.userSteps.getDarkSideSteps().getDirectJooqDbSteps().useShardForLogin(CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT)
                .campaignsSteps().setCampaignsStatusModerate(campaignId, CampaignsStatusmoderate.Yes);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void setAdStatuses() {
        adImageHash = api.userSteps.imagesStepsV5().addImageAdAdImage(5);
        adId = api.userSteps.adsSteps().addDefaultTextImageAd(adGroupId, adImageHash);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(adId);
        dbSteps.imagesSteps().setImageStatusModerate(adId, adImageHash, ImagesStatusmoderate.Yes);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adId, Status.YES);
        api.userSteps.bannersFakeSteps().setLastChange(adId, previousLastChange.toString());
    }

    @Test
    @Description("Проверка обновления статусов при обновлении Href")
    public void updateHref() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withHref(TextAdAddMap.DEFAULT_HREF + "/update")
                        ).withId(adId)),
                ExpectedResult.success()
        );

        BannersRecord bannersRecord = dbSteps.bannersSteps().getBanner(adId);
        assertThat("в БД верное значение StatusBsSynced",
                bannersRecord.getStatusbssynced(), equalTo(BannersStatusbssynced.No));
        assertThat("в БД верное значение StatusModerate",
                bannersRecord.getStatusmoderate(), anyOf(
                        equalTo(BannersStatusmoderate.Ready),
                        equalTo(BannersStatusmoderate.Sending)
                ));
        assertThat("в БД верное значение LastChange",
                bannersRecord.getLastchange().toLocalDateTime(), greaterThan(previousLastChange));

        ImagesRecord image = dbSteps.imagesSteps().getImagesRecords(campaignId, adGroupId, adId);
        assertThat("в БД сбросился StatusImageModerate",
                image.getStatusmoderate(), equalTo(ImagesStatusmoderate.Ready));
    }

    @Test
    @Description("Проверка обновления статусов при изменении изображения в объявлении")
    public void updateAdImageHash() {
        String newAdImageHash = api.userSteps.imagesStepsV5().addImageAdAdImage(6);
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withAdImageHash(newAdImageHash)
                        ).withId(adId)),
                ExpectedResult.success()
        );

        BannersRecord bannersRecord = dbSteps.bannersSteps().getBanner(adId);
        assertThat("в БД верное значение StatusBsSynced",
                bannersRecord.getStatusbssynced(), equalTo(BannersStatusbssynced.No));
        assertThat("в БД верное значение StatusModerate",
                bannersRecord.getStatusmoderate(), anyOf(
                        equalTo(BannersStatusmoderate.Ready),
                        equalTo(BannersStatusmoderate.Sending)
                ));
        assertThat("в БД верное значение LastChange",
                bannersRecord.getLastchange().toLocalDateTime(), greaterThan(previousLastChange));
        ImagesRecord image = dbSteps.imagesSteps().getImagesRecords(campaignId, adGroupId, adId);
        assertThat("в БД сбросился StatusImageModerate",
                image.getStatusmoderate(), equalTo(ImagesStatusmoderate.Ready));
    }


    @Test
    @Description("Проверка обновления статусов при изменении изображения в объявлении на такое же")
    public void updateAdImageHashSame() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withAdImageHash(adImageHash)
                        ).withId(adId)),
                ExpectedResult.success()
        );

        BannersRecord bannersRecord = dbSteps.bannersSteps().getBanner(adId);
        assertThat("в БД верное значение StatusBsSynced",
                bannersRecord.getStatusbssynced(), equalTo(BannersStatusbssynced.Yes));
        assertThat("в БД верное значение StatusModerate",
                bannersRecord.getStatusmoderate(), equalTo(BannersStatusmoderate.Yes));
        assertThat("в БД верное значение LastChange",
                bannersRecord.getLastchange().toLocalDateTime(), equalTo(previousLastChange));

        ImagesRecord image = dbSteps.imagesSteps().getImagesRecords(campaignId, adGroupId, adId);
        assertThat("в БД сбросился StatusBannerImageModerate",
                image.getStatusmoderate(), equalTo(ImagesStatusmoderate.Yes));
    }


    @Test
    @Description("Проверка необновления статусов при неудавшемся обновлении")
    public void updateWithError() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withTextImageAd(
                                new TextImageAdUpdateMap().withHref("12345")
                        ).withId(adId)),
                ExpectedResult.errors(
                        new Notification(5005, Api5ErrorDetails.INCORRECT_LINK)
                )
        );

        BannersRecord bannersRecord = dbSteps.bannersSteps().getBanner(adId);
        assertThat("в БД верное значение StatusBsSynced",
                bannersRecord.getStatusbssynced(), equalTo(BannersStatusbssynced.Yes));
        assertThat("в БД верное значение StatusModerate",
                bannersRecord.getStatusmoderate(), equalTo(BannersStatusmoderate.Yes));
        assertThat("в БД верное значение LastChange",
                bannersRecord.getLastchange().toLocalDateTime(), equalTo(previousLastChange));
    }
}
