package ru.yandex.autotests.directapi.agencyclients.add;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.List;
import java.util.function.IntFunction;

import com.yandex.direct.api.v5.agencyclients.AddResponse;
import com.yandex.direct.api.v5.agencyclients.AgencyClientFieldEnum;
import com.yandex.direct.api.v5.agencyclients.GetResponse;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.generalclients.ClientGetItem;
import com.yandex.direct.api.v5.generalclients.PrivilegeEnum;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsFeatures;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5;
import ru.yandex.autotests.directapi.model.api5.agencyclients.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.agencyclients.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.ClientGetItemMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.GrantGetItemMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.GrantItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.hasItems;
import static org.hamcrest.Matchers.hasProperty;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(AgencyClientsFeatures.ADD)
@Description("Проверка добавления с выставлением грантов")
public class AddGrantsTest {
    private static final String AGENCY_LOGIN = AgencyClientsLogins.AGENCY_RUB;
    private static final String SUBCLIENT_LOGIN_BASE = "ac-add-grants-cl";
    public static final String AGENCY_NAME = "Тестовое агентство для API5 AgencyClients";

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(AGENCY_LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().enableToCreateSubClients(AGENCY_LOGIN);
    }

    @Test
    public void addSubclientWithDefaultsAllGrantsOff() {
        String login = AddRequestMap.generateLogin(SUBCLIENT_LOGIN_BASE);
        AddResponse response =
                api.userSteps.agencyClientsStepsV5().agencyClientsAdd(new AddRequestMap().withDefault(login));
        assertThat("добавление субклиента с набором грантов прошло успешно", response,
                hasProperty("clientId", not(nullValue())));
        checkSwitchedOnSubclientGrants(login);
    }

    @Test
    public void addSubclientAllGrantsOff() {
        String login = AddRequestMap.generateLogin(SUBCLIENT_LOGIN_BASE);
        api.userSteps.agencyClientsStepsV5().agencyClientsAdd(
                new AddRequestMap().withDefault(login).withGrants(
                        new GrantItemMap(PrivilegeEnum.TRANSFER_MONEY, YesNoEnum.NO),
                        new GrantItemMap(PrivilegeEnum.IMPORT_XLS, YesNoEnum.NO),
                        new GrantItemMap(PrivilegeEnum.EDIT_CAMPAIGNS, YesNoEnum.NO)
                )
        );
        checkSwitchedOnSubclientGrants(login);
    }

    @Test
    public void addSubclientAllGrantsOn() {
        String login = AddRequestMap.generateLogin(SUBCLIENT_LOGIN_BASE);
        api.userSteps.agencyClientsStepsV5().agencyClientsAdd(
                new AddRequestMap().withDefault(login).withGrants(
                        new GrantItemMap(PrivilegeEnum.TRANSFER_MONEY, YesNoEnum.YES),
                        new GrantItemMap(PrivilegeEnum.IMPORT_XLS, YesNoEnum.YES),
                        new GrantItemMap(PrivilegeEnum.EDIT_CAMPAIGNS, YesNoEnum.YES)
                )
        );
        checkSwitchedOnSubclientGrants(login, PrivilegeEnum.TRANSFER_MONEY, PrivilegeEnum.IMPORT_XLS,
                PrivilegeEnum.EDIT_CAMPAIGNS);
    }

    @Test
    public void switchingOnEditCampaignAndSwitchingOffImportXlsIsOk() {
        String login = AddRequestMap.generateLogin(SUBCLIENT_LOGIN_BASE);
        api.userSteps.agencyClientsStepsV5().agencyClientsAdd(
                new AddRequestMap().withDefault(login).withGrants(
                        new GrantItemMap(PrivilegeEnum.EDIT_CAMPAIGNS, YesNoEnum.YES),
                        new GrantItemMap(PrivilegeEnum.IMPORT_XLS, YesNoEnum.NO)
                )
        );
        checkSwitchedOnSubclientGrants(login, PrivilegeEnum.EDIT_CAMPAIGNS);
    }

    @Test
    public void switchingOffEditCampaignAndSwitchingOnImportXlsCauseAnError() {
        String login = AddRequestMap.generateLogin(SUBCLIENT_LOGIN_BASE);
        api.userSteps.agencyClientsStepsV5().expectErrorOnAgencyClientsAdd(
                new AddRequestMap().withDefault(login).withGrants(
                        new GrantItemMap(PrivilegeEnum.EDIT_CAMPAIGNS, YesNoEnum.NO),
                        new GrantItemMap(PrivilegeEnum.IMPORT_XLS, YesNoEnum.YES)
                ),
                new Api5Error(6000, Api5ErrorDetailsJava.INCONSISTENT_STATE_ALLOW_EDIT_CAMPAIGN_AND_ALLOW_IMPORT_XLS)
        );
    }

    private void checkSwitchedOnSubclientGrants(String login, PrivilegeEnum... switchedOnPrivileges) {
        GetResponse getResponse = api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(login).withFieldNames(AgencyClientFieldEnum.GRANTS));
        List<ClientGetItem> actualResult = getResponse.getClients();

        EnumSet<PrivilegeEnum> switchedOn = EnumSet.noneOf(PrivilegeEnum.class);
        switchedOn.addAll(Arrays.asList(switchedOnPrivileges));

        BeanDifferMatcher<GrantGetItemMap>[] grantsMatchers = EnumSet.allOf(PrivilegeEnum.class).stream()
                .map(p -> new GrantGetItemMap(
                        AGENCY_NAME, p, switchedOn.contains(p) ? YesNoEnum.YES : YesNoEnum.NO).getBean())
                .map(BeanDifferMatcherV5::beanDifferV5)
                .toArray((IntFunction<BeanDifferMatcher<GrantGetItemMap>[]>) BeanDifferMatcher[]::new);

        assumeThat("гранты в ожидаемом состоянии",
                actualResult, contains(hasProperty(ClientGetItemMap.GRANTS, hasItems(grantsMatchers))));
    }
}
