package ru.yandex.autotests.directapi.agencyclients.add;

import java.util.Arrays;
import java.util.Collection;

import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsFeatures;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.ClientFakeInfo;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.agencyclients.AddRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

@Aqua.Test
@Features(AgencyClientsFeatures.ADD)
@Description("Проверка прав доступа, негативные сценарии")
@Issue("https://st.yandex-team.ru/DIRECT-65272")
@RunWith(Parameterized.class)
public class CheckRightsNegativeTest {
    public static final String LOGIN_PREFIX = "agency-clients-role-neg";
    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public String operatorLogin;

    @Parameterized.Parameter(2)
    public String agencyLogin;

    @Parameterized.Parameter(3)
    public Boolean canCreateSubclients;

    @Parameterized.Parameter(4)
    public Api5Error expectedError;
    private String prevAllowCreateSubclients;

    @Parameterized.Parameters(name = "{0} (operatorLogin = {1}, agencyLogin = {2}, canCreateSubclients = {3}, expectedErrorDetail = {4})")
    public static Collection<Object[]> testData() {
        return Arrays.asList(new Object[][]{
                {"В Client-Login может быть только представитель агентства",
                        AgencyClientsLogins.AGENCY_RUB, AgencyClientsLogins.AGENCY_RUB_SUBCLIENT, null,
                        error(8000, Api5ErrorDetails.CLIENT_LOGIN_MUST_BE_AGENCY)},
                {"Агентство, которому не разрешено создавать субклиентов, получает ошибку авторизации",
                        AgencyClientsLogins.AGENCY_WITH_NORIGHT_TO_CREATE_SUBCLIENT,
                        AgencyClientsLogins.AGENCY_WITH_NORIGHT_TO_CREATE_SUBCLIENT, Boolean.FALSE,
                        error(3001, Api5ErrorDetails.NO_RIGHTS_TO_CREATE_CLIENTS)},
                {"Пользователь с ролью SUPER_READER не может пользоваться этим сервисом",
                        Logins.SUPER_READER, AgencyClientsLogins.AGENCY_RUB, null,
                        error(54, Api5ErrorDetails.EMPTY_STRING)},
                {"Пользователь с ролью SUPPORT не может пользоваться этим сервисом",
                        Logins.SUPPORT, AgencyClientsLogins.AGENCY_RUB, null,
                        error(54, Api5ErrorDetails.EMPTY_STRING)},
                {"Пользователь с ролью MEDIA не может пользоваться этим сервисом",
                        Logins.MEDIA, AgencyClientsLogins.AGENCY_RUB, null,
                        error(54, Api5ErrorDetails.EMPTY_STRING)},
                {"Пользователь с ролью PLACER не может пользоваться этим сервисом",
                        Logins.PLACER, AgencyClientsLogins.AGENCY_RUB, null,
                        error(54, Api5ErrorDetails.EMPTY_STRING)},
        });
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void before() {
        api.as(operatorLogin);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(operatorLogin);
        ClientFakeInfo clientData = api.userSteps.clientFakeSteps().getClientData(agencyLogin);
        if (canCreateSubclients != null) {
            prevAllowCreateSubclients = clientData.getAllowCreateSubclients();
            if (canCreateSubclients) {
                api.userSteps.clientFakeSteps().enableToCreateSubClients(agencyLogin);
            } else {
                api.userSteps.clientFakeSteps().setAgencyCreateSubClients(agencyLogin, "No");
            }
        } else {
            prevAllowCreateSubclients = null;
        }
    }

    @After
    public void after() {
        if (prevAllowCreateSubclients != null) {
            api.userSteps.clientFakeSteps().setAgencyCreateSubClients(agencyLogin, prevAllowCreateSubclients);
        }
    }

    @Test
    public void rolesTest() {
        api.userSteps.agencyClientsStepsV5().expectErrorOnAgencyClientsAdd(
                new AddRequestMap().withDefault(AddRequestMap.generateLogin(LOGIN_PREFIX)),
                agencyLogin,
                expectedError
        );
    }

    private static Api5Error error(int code, Api5ErrorDetails detail) {
        return new Api5Error(code, detail);
    }
}
