package ru.yandex.autotests.directapi.agencyclients.get;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.agencyclients.AgencyClientFieldEnum;
import com.yandex.direct.api.v5.agencyclients.GetResponse;
import com.yandex.direct.api.v5.general.LangEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.generalclients.ClientGetItem;
import com.yandex.direct.api.v5.generalclients.EmailSubscriptionEnum;
import com.yandex.direct.api.v5.generalclients.NotificationGet;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.UsersRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsFeatures;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsLogins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.agencyclients.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.EmailSubscriptionItemMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.NotificationGetMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

@Aqua.Test
@Features(AgencyClientsFeatures.GET)
@Description("Проверка получения Notification")
public class GetNotificationTest {
    private static final String AGENCY_LOGIN = AgencyClientsLogins.AGENCY_RUB;
    private static final String SUBCLIENT1_LOGIN = AgencyClientsLogins.AGENCY_RUB_SUBCLIENT3_NOTIFICATIONS;
    private static final String SUBCLIENT2_LOGIN = AgencyClientsLogins.AGENCY_RUB_SUBCLIENT2_WITH_PHONE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(AGENCY_LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private List<Object> expectedNotifications = new ArrayList<>();

    @Before
    @Step("Подготовка тестовых данных")
    public void prepare() {
        DirectJooqDbSteps dbSteps = api.userSteps.getDirectJooqDbSteps();

        dbSteps.useShardForLogin(SUBCLIENT1_LOGIN);
        UsersRecord usersRecord = dbSteps.usersSteps().getUser(SUBCLIENT1_LOGIN);
        NotificationGetMap notificationMap = new NotificationGetMap()
                .withLang(LangEnum.fromValue(usersRecord.getLang().toUpperCase()))
                .withEmail(usersRecord.getEmail())
                .withSmsPhoneNumber("")
                .withEmailSubscriptions(
                        new EmailSubscriptionItemMap(EmailSubscriptionEnum.RECEIVE_RECOMMENDATIONS,
                                YesNoEnum.fromValue(usersRecord.getSendnews().getLiteral().toUpperCase())),
                        new EmailSubscriptionItemMap(EmailSubscriptionEnum.TRACK_MANAGED_CAMPAIGNS,
                                YesNoEnum.fromValue(usersRecord.getSendaccnews().getLiteral().toUpperCase())),
                        new EmailSubscriptionItemMap(EmailSubscriptionEnum.TRACK_POSITION_CHANGES,
                                YesNoEnum.fromValue(usersRecord.getSendwarn().getLiteral().toUpperCase()))
                );
        expectedNotifications.add(notificationMap.getBean());

        dbSteps.useShardForLogin(SUBCLIENT2_LOGIN);
        usersRecord = dbSteps.usersSteps().getUser(SUBCLIENT2_LOGIN);
        notificationMap = new NotificationGetMap()
                .withLang(LangEnum.fromValue(usersRecord.getLang().toUpperCase()))
                .withEmail(usersRecord.getEmail())
                .withSmsPhoneNumber("+7000*****76")
                .withEmailSubscriptions(
                        new EmailSubscriptionItemMap(EmailSubscriptionEnum.RECEIVE_RECOMMENDATIONS,
                                YesNoEnum.fromValue(usersRecord.getSendnews().getLiteral().toUpperCase())),
                        new EmailSubscriptionItemMap(EmailSubscriptionEnum.TRACK_MANAGED_CAMPAIGNS,
                                YesNoEnum.fromValue(usersRecord.getSendaccnews().getLiteral().toUpperCase())),
                        new EmailSubscriptionItemMap(EmailSubscriptionEnum.TRACK_POSITION_CHANGES,
                                YesNoEnum.fromValue(usersRecord.getSendwarn().getLiteral().toUpperCase())));
        expectedNotifications.add(notificationMap.getBean());

        expectedNotifications.sort(Comparator.comparing(e -> ((NotificationGet) e).getEmail()));

        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AGENCY_LOGIN);
    }

    @Test
    public void getNotificationTest() {
        GetResponse getResponse = api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(SUBCLIENT1_LOGIN, SUBCLIENT2_LOGIN)
                        .withFieldNames(AgencyClientFieldEnum.NOTIFICATION));

        List<NotificationGet> actualNotification = getResponse.getClients().stream().map(ClientGetItem::getNotification)
                .sorted(Comparator.comparing(NotificationGet::getEmail)).collect(Collectors.toList());

        assertThat("полученные поля совпадают с ожидаемыми", actualNotification, beanDifferV5(expectedNotifications));
    }
}
