package ru.yandex.autotests.directapi.agencyclients.get;

import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.agencyclients.AgencyClientFieldEnum;
import com.yandex.direct.api.v5.agencyclients.GetResponse;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsFeatures;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.balanceclient.NotifyClient2JSONRequest;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.ClientFakeInfo;
import ru.yandex.autotests.directapi.darkside.steps.fakesteps.ClientFakeSteps;
import ru.yandex.autotests.directapi.model.api5.agencyclients.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.IsEqual.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(AgencyClientsFeatures.GET)
@Description(("Проверка получения OverdraftSumAvailable"))
@RunWith(Parameterized.class)
public class GetOverdraftTest {
    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter()
    public String agencyLogin;

    @Parameterized.Parameter(1)
    public String subclientLogin;

    @Parameterized.Parameter(2)
    public Currency currency;

    protected LogSteps log = LogSteps.getLogger(this.getClass());

    private Long expectedOverdraft;

    @Parameterized.Parameters(name = "Агентство {0}, cубклиент {1}, валюта {2}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
                // NB: я не смог найти способа создать нового субклиента в уе - кажется, все способы уже закрыли.
                // Существующих клиентов не хочется использовать потому, что в тесте изменяются данные такого
                // клиента и есть шанс этим испортить другие тесты
                {AgencyClientsLogins.AGENCY_RUB, AgencyClientsLogins.AGENCY_RUB_SUBCLIENT, Currency.RUB},
                {AgencyClientsLogins.AGENCY_UAH, AgencyClientsLogins.AGENCY_UAH_SUBCLIENT, Currency.UAH},
                {AgencyClientsLogins.AGENCY_EUR, AgencyClientsLogins.AGENCY_EUR_SUBCLIENT, Currency.EUR},
        });
    }

    private double convertToPercents(double value) {
        return value * 100;
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void prepare() {
        MoneyCurrency moneyCurrency = MoneyCurrency.get(currency);

        Money overdraftLimit = Money.valueOf(moneyCurrency.getMinTransferAmount().floatValue(), currency);
        expectedOverdraft = overdraftLimit.subtractVAT().setScale(2, RoundingMode.DOWN).bidLong().longValue();

        ClientFakeSteps clientFakeSteps = api.userSteps.clientFakeSteps();

        ClientFakeInfo fakeInfo = clientFakeSteps.getClientData(subclientLogin);

        log.info("Вызываем метод NotifyClient2 - выставим значение овердрафта субклиенту " + subclientLogin);
        NotifyClient2JSONRequest request = new NotifyClient2JSONRequest().defaultParams()
                .withClientID(Long.valueOf(fakeInfo.getClientID()))
                .withTimestamp()
                .withOverdraftLimit(overdraftLimit.floatValue())
                .withOverdraftSpent(0f)
                .withClientCurrency(currency.value());
        api.userSteps.getDarkSideSteps().getBalanceClientNotifyClientJsonSteps().notifyClientNoErrors(request);

        if (moneyCurrency.isVATIncludedInRate()) {
            log.info(String.format("Выставим значение %.2f НДС агентству %s с валютой %s", moneyCurrency.getVatRate(),
                    agencyLogin, currency));
            clientFakeSteps.setVATRate(agencyLogin,
                    convertToPercents(moneyCurrency.getVatRate())); // convert vat rate to percent as db needs
        }

        clientFakeSteps.fakeClearClientSpentUnits(agencyLogin);
    }

    @Test
    public void getOverdraftTest() {
        api.as(agencyLogin);

        GetResponse response = api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(subclientLogin)
                        .withFieldNames(AgencyClientFieldEnum.OVERDRAFT_SUM_AVAILABLE));

        assumeThat("полученный список с данными 1 клиента", response.getClients(), hasSize(1));

        assertThat("полученное значение совпадает с ожидаемым",
                response.getClients().get(0).getOverdraftSumAvailable(), equalTo(expectedOverdraft));
    }
}
