package ru.yandex.autotests.directapi.agencyclients.get;

import java.util.Collections;

import com.yandex.direct.api.v5.agencyclients.AgencyClientFieldEnum;
import com.yandex.direct.api.v5.agencyclients.GetResponse;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsFeatures;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsLogins;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.agencyclients.AgencyClientsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.agencyclients.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.ClientGetItemMap;
import ru.yandex.autotests.directapi.model.api5.general.LimitOffsetMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.MatcherAssert.assertThat;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;

@Aqua.Test
@Features(AgencyClientsFeatures.GET)
@Description("Проверка Page")
public class GetPageTest {
    private static final String AGENCY_LOGIN = AgencyClientsLogins.AGENCY_RUB;
    private static final String SUBCLIENT_LOGIN = AgencyClientsLogins.AGENCY_RUB_SUBCLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(AGENCY_LOGIN).locale("RU/ru");

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Before
    @Step("Подготовка тестовых данных")
    public void prepare() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AGENCY_LOGIN);
    }

    @Test
    public void getPageWithLimitLessThanMinAllowedNegativeTest() {
        api.userSteps.agencyClientsStepsV5().expectErrorOnAgencyClientsGet(
                new GetRequestMap()
                        .withSelectionCriteria(new AgencyClientsSelectionCriteriaMap().withLogins(SUBCLIENT_LOGIN))
                        .withFieldNames(AgencyClientFieldEnum.LOGIN)
                        .withPage(new LimitOffsetMap().withLimit(0L)),
                new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0));
    }

    @Test
    public void getPageWithLimitGreaterThanMaxAllowedNegativeTest() {
        api.userSteps.agencyClientsStepsV5().expectErrorOnAgencyClientsGet(
                new GetRequestMap()
                        .withSelectionCriteria(new AgencyClientsSelectionCriteriaMap().withLogins(SUBCLIENT_LOGIN))
                        .withFieldNames(AgencyClientFieldEnum.LOGIN)
                        .withPage(new LimitOffsetMap().withLimit(LimitOffsetMap.LIMIT_MAX_LENGTH + 1)),
                new Api5Error(4002, Api5ErrorDetails.LIMIT_CANNOT_BE_MORE, LimitOffsetMap.LIMIT_MAX_LENGTH));
    }

    @Test
    public void getPageWithOffsetLessThanMinAllowedNegativeTest() {
        api.userSteps.agencyClientsStepsV5().expectErrorOnAgencyClientsGet(
                new GetRequestMap()
                        .withSelectionCriteria(new AgencyClientsSelectionCriteriaMap().withLogins(SUBCLIENT_LOGIN))
                        .withFieldNames(AgencyClientFieldEnum.LOGIN)
                        .withPage(new LimitOffsetMap().withOffset(-1L)),
                new Api5Error(4002, Api5ErrorDetails.OFFSET_MUST_BE_MORE_OR_EQUAL_THAN_0));
    }

    @Test
    public void getPageWithOffsetGreaterThanMaxAllowedNegativeTest() {
        api.userSteps.agencyClientsStepsV5().expectErrorOnAgencyClientsGet(
                new GetRequestMap()
                        .withSelectionCriteria(new AgencyClientsSelectionCriteriaMap().withLogins(SUBCLIENT_LOGIN))
                        .withFieldNames(AgencyClientFieldEnum.LOGIN)
                        .withPage(new LimitOffsetMap().withOffset(LimitOffsetMap.OFFSET_MAX_LENGTH + 1)),
                new Api5Error(4002, Api5ErrorDetails.OFFSET_MUST_BE_LESS_OR_EQUAL_THAN,
                        LimitOffsetMap.OFFSET_MAX_LENGTH));
    }

    @Test
    public void getPagePositiveTest() {
        GetResponse getResponse = api.userSteps.agencyClientsStepsV5().agencyClientsGet(new GetRequestMap()
                .withSelectionCriteria(new AgencyClientsSelectionCriteriaMap().withLogins(SUBCLIENT_LOGIN))
                .withFieldNames(AgencyClientFieldEnum.LOGIN)
                .withPage(new LimitOffsetMap().withOffset(0L).withLimit(1L)));

        assertThat("получен корректный ответ", getResponse.getClients(),
                beanDifferV5(Collections.singletonList(new ClientGetItemMap().withLogin(SUBCLIENT_LOGIN).getBean())));
    }
}