package ru.yandex.autotests.directapi.agencyclients.get;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

import com.yandex.direct.api.v5.agencyclients.AgencyClientFieldEnum;
import com.yandex.direct.api.v5.agencyclients.GetResponse;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.generalclients.ClientGetItem;
import com.yandex.direct.api.v5.generalclients.ClientSettingGetEnum;
import com.yandex.direct.api.v5.generalclients.ClientSettingGetItem;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.ClientsOptionsRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.UsersRecord;
import ru.yandex.autotests.direct.db.steps.ClientsOptionsSteps;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsFeatures;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsLogins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.agencyclients.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.ClientGetItemMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.ClientSettingGetItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(AgencyClientsFeatures.GET)
@Description("Проверка получения Settings")
public class GetSettingsTest {
    private static final String AGENCY_LOGIN = AgencyClientsLogins.AGENCY_RUB;
    private static final String SUBCLIENT_LOGIN_WITH_ACCOUNT = AgencyClientsLogins.AGENCY_RUB_SUBCLIENT;
    private static final String SUBCLIENT_LOGIN_WITHOUT_ACCOUNT = AgencyClientsLogins.AGENCY_RUB_SUBCLIENT1;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(AGENCY_LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private List<Object> expectedResult;

    @BeforeClass
    @Step("Подготовка и проверка тестовых данных перед всеми тестами")
    public static void commonPrepare() {
        // включаем общий счёт для первого субклиента
        api.userSteps.financeSteps().verifySharedAccount(SUBCLIENT_LOGIN_WITH_ACCOUNT);
        assumeThat("общий счёт подключен", api.userSteps.financeSteps().getAccount(SUBCLIENT_LOGIN_WITH_ACCOUNT),
                notNullValue());

        // проверяем, что у второго субклиента нет общего счета
        assumeThat("общий счёт не подключен", api.userSteps.financeSteps().getAccount(SUBCLIENT_LOGIN_WITHOUT_ACCOUNT),
                nullValue());
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void prepare() {
        expectedResult = new ArrayList<>();

        DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps();

        jooqDbSteps.useShardForLogin(SUBCLIENT_LOGIN_WITH_ACCOUNT);
        UsersRecord usersRecord = jooqDbSteps.usersSteps().getUser(SUBCLIENT_LOGIN_WITH_ACCOUNT);
        ClientsOptionsRecord clientsOptionsRecord = new ClientsOptionsRecord()
                .setClientid(usersRecord.getClientid())
                .setHideMarketRating(0)
                .setClientFlags("");
        jooqDbSteps.clientsOptionsSteps().updateClientOptions(clientsOptionsRecord);

        expectedResult.add(
                new ClientGetItemMap().withSettings(
                        new ClientSettingGetItemMap().withOption(ClientSettingGetEnum.SHARED_ACCOUNT_ENABLED)
                                .withValue(YesNoEnum.YES),
                        new ClientSettingGetItemMap().withOption(ClientSettingGetEnum.DISPLAY_STORE_RATING)
                                .withValue(YesNoEnum.YES),
                        new ClientSettingGetItemMap().withOption(ClientSettingGetEnum.CORRECT_TYPOS_AUTOMATICALLY)
                                .withValue(YesNoEnum.YES)).getBean());

        jooqDbSteps.useShardForLogin(SUBCLIENT_LOGIN_WITHOUT_ACCOUNT);
        usersRecord = jooqDbSteps.usersSteps().getUser(SUBCLIENT_LOGIN_WITHOUT_ACCOUNT);
        clientsOptionsRecord = new ClientsOptionsRecord()
                .setClientid(usersRecord.getClientid())
                .setHideMarketRating(1)
                .setClientFlags(ClientsOptionsSteps.ClientFlagsEnum.NO_TEXT_AUTOCORRECTION.toString());
        jooqDbSteps.clientsOptionsSteps().updateClientOptions(clientsOptionsRecord);

        expectedResult.add(
                new ClientGetItemMap().withSettings(
                        new ClientSettingGetItemMap().withOption(ClientSettingGetEnum.SHARED_ACCOUNT_ENABLED)
                                .withValue(YesNoEnum.NO),
                        new ClientSettingGetItemMap().withOption(ClientSettingGetEnum.DISPLAY_STORE_RATING)
                                .withValue(YesNoEnum.NO),
                        new ClientSettingGetItemMap().withOption(ClientSettingGetEnum.CORRECT_TYPOS_AUTOMATICALLY)
                                .withValue(YesNoEnum.NO)).getBean());

        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AGENCY_LOGIN);
    }

    @Test
    public void getSettingsTest() {
        GetResponse getResponse = api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(SUBCLIENT_LOGIN_WITH_ACCOUNT, SUBCLIENT_LOGIN_WITHOUT_ACCOUNT)
                        .withFieldNames(AgencyClientFieldEnum.SETTINGS));

        List<ClientGetItem> actualResult = getResponse.getClients();

        actualResult.forEach(e -> e.getSettings().sort(Comparator.comparing(ClientSettingGetItem::getOption)));

        expectedResult.forEach(
                e -> ((ClientGetItem) e).getSettings().sort(Comparator.comparing(ClientSettingGetItem::getOption)));

        assertThat("полученные значения совпадают с ожидаемыми", actualResult, beanDifferV5(expectedResult));
    }
}
