package ru.yandex.autotests.directapi.agencyclients.get;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.yandex.direct.api.v5.agencyclients.AgencyClientFieldEnum;
import com.yandex.direct.api.v5.agencyclients.GetResponse;
import com.yandex.direct.api.v5.general.CurrencyEnum;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.AccountScoreRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.ClientsRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.UsersRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsFeatures;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.agencyclients.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.ClientGetItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

@Aqua.Test
@Features(AgencyClientsFeatures.GET)
@Description("Проверка получения данных нескольких клиентов за одни запрос")
public class GetSeveralClientsTest {
    private static final String PATTERN = "yyyy-MM-dd";
    private static final String AGENCY_LOGIN = AgencyClientsLogins.AGENCY_RUB;
    private static final String SUBCLIENT1 = AgencyClientsLogins.AGENCY_RUB_SUBCLIENT;
    private static final String SUBCLIENT2 = AgencyClientsLogins.AGENCY_RUB_SUBCLIENT1;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(5).as(AGENCY_LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private List<Object> expectedResults;
    private AgencyClientFieldEnum[] simpleFields = new AgencyClientFieldEnum[]{
            AgencyClientFieldEnum.LOGIN,
            AgencyClientFieldEnum.CLIENT_ID,
            AgencyClientFieldEnum.CLIENT_INFO,
            AgencyClientFieldEnum.COUNTRY_ID,
            AgencyClientFieldEnum.PHONE,
            AgencyClientFieldEnum.CURRENCY,
            AgencyClientFieldEnum.CREATED_AT,
            AgencyClientFieldEnum.ACCOUNT_QUALITY
    };

    @Before
    @Step("Подготовка тестовых данных")
    public void prepare() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AGENCY_LOGIN);

        expectedResults = new ArrayList<>();
        for (String subclient : Arrays.asList(SUBCLIENT1, SUBCLIENT2)) {
            DirectJooqDbSteps dbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(subclient);

            UsersRecord usersRecord = dbSteps.usersSteps().getUser(subclient);

            ClientsRecord clientsRecord = dbSteps.clientsSteps().getClients(usersRecord.getClientid());

            AccountScoreRecord accountScoreRecord =
                    dbSteps.accountScoreSteps().getLastAccountScore(usersRecord.getClientid());

            BigDecimal expectedAccountScore =
                    accountScoreRecord == null ? null : accountScoreRecord.getScore().stripTrailingZeros();

            String expectedCreatedAt = DateTimeFormat.forPattern(PATTERN)
                    .print(new DateTime(usersRecord.getCreatetime() * 1000));

            CurrencyEnum expectedCurrency = clientsRecord.getWorkCurrency() == null ? CurrencyEnum.YND_FIXED
                    : CurrencyEnum.fromValue(clientsRecord.getWorkCurrency().getLiteral());

            Long countryId = clientsRecord.getCountryRegionId();

            String phone = usersRecord.getPhone();

            expectedResults.add(
                    new ClientGetItemMap()
                            .withLogin(subclient)
                            .withClientId(usersRecord.getClientid())
                            .withCreatedAt(expectedCreatedAt)
                            .withCurrency(expectedCurrency)
                            .withPhone(phone == null ? "" : phone)
                            .withAccountQuality(expectedAccountScore)
                            .withClientInfo(usersRecord.getFio())
                            .withCountryId(countryId.intValue())
                            .getBean());
        }
    }

    @Test
    public void getSeveralClientsAtOnceTest() {
        GetResponse getResponse = api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(SUBCLIENT1, SUBCLIENT2).withFieldNames(simpleFields));

        assertThat("полученные поля совпадают с ожидаемыми", getResponse.getClients(), beanDifferV5(expectedResults));
    }
}
