package ru.yandex.autotests.directapi.agencyclients.get;

import com.yandex.direct.api.v5.agencyclients.AgencyClientFieldEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsFeatures;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsLogins;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.agencyclients.AgencyClientsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.agencyclients.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.LimitOffsetMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.CoreMatchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

@Aqua.Test
@Features(AgencyClientsFeatures.GET)
@Description("Проверка списания баллов")
public class GetUnitsTest {
    private static final String AGENCY_LOGIN = AgencyClientsLogins.AGENCY_UNITS;
    private static final String SUBCLIENT1_LOGIN = AgencyClientsLogins.AGENCY_UNITS_SUBCLIENT;
    private static final String SUBCLIENT2_LOGIN = AgencyClientsLogins.AGENCY_UNITS_SUBCLIENT2;
    private static final String OTHER_AGENCY_SUBCLIENT = AgencyClientsLogins.OTHER_AGENCY_SUBCLIENT;

    private static final int CALL_COST = 10;
    private static final int PER_OBJECT_COST = 1;
    private static final int COMMON_REQUEST_ERROR = 50;
    private static final int DEFAULT_ERROR_COST = 20;

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(AGENCY_LOGIN);

    private int unitsBefore;

    @Before
    @Step("Подготовка тестовых данных")
    public void prepare() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AGENCY_LOGIN);
        unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);
    }

    @Test
    public void getNoClientsFoundTest() {
        // NB: агентство не может запрашивать данные чужого субклиента и поэтому запрос ничего не вернет
        api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(OTHER_AGENCY_SUBCLIENT)
                        .withFieldNames(AgencyClientFieldEnum.CLIENT_ID));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);

        assertThat("списалось правильное количество баллов", unitsBefore - unitsAfter, equalTo(CALL_COST));
    }

    @Test
    public void getOneClientWithOneFieldTest() {
        api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(SUBCLIENT1_LOGIN)
                        .withFieldNames(AgencyClientFieldEnum.CLIENT_ID));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);

        assertThat("списалось правильное количество баллов", unitsBefore - unitsAfter,
                equalTo(CALL_COST + PER_OBJECT_COST));
    }

    @Test
    public void getOneClientWithSeveralFieldsTest() {
        api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(SUBCLIENT1_LOGIN)
                        .withFieldNames(AgencyClientFieldEnum.CLIENT_ID, AgencyClientFieldEnum.LOGIN));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);

        assertThat("списалось правильное количество баллов", unitsBefore - unitsAfter,
                equalTo(CALL_COST + PER_OBJECT_COST));
    }

    @Test
    public void getSeveralClientsWithOneFieldTest() {
        String[] subclients = new String[]{SUBCLIENT1_LOGIN, SUBCLIENT2_LOGIN};

        api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(subclients).withFieldNames(AgencyClientFieldEnum.CLIENT_ID));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);

        assertThat("списалось правильное количество баллов", unitsBefore - unitsAfter,
                equalTo(CALL_COST + subclients.length * PER_OBJECT_COST));
    }

    @Test
    public void getSeveralClientsWithSeveralFieldsTest() {
        String[] subclients = new String[]{SUBCLIENT1_LOGIN, SUBCLIENT2_LOGIN};

        api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(subclients)
                        .withFieldNames(AgencyClientFieldEnum.CLIENT_ID, AgencyClientFieldEnum.LOGIN));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);

        assertThat("списалось правильное количество баллов", unitsBefore - unitsAfter,
                equalTo(CALL_COST + subclients.length * PER_OBJECT_COST));
    }

    @Test
    public void getWithIncorrectCommonRequestTest() {
        api.userSteps.agencyClientsStepsV5().expectErrorOnAgencyClientsGet(
                new GetRequestMap().withAllFieldNames(),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        capitalize(GetRequestMap.SELECTION_CRITERIA)));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);

        assertThat("списалось правильное количество баллов", unitsBefore - unitsAfter, equalTo(COMMON_REQUEST_ERROR));
    }

    @Test
    public void getWithIncorrectRequestTest() {
        api.userSteps.agencyClientsStepsV5().expectErrorOnAgencyClientsGet(
                new GetRequestMap()
                        .withSelectionCriteria(new AgencyClientsSelectionCriteriaMap().withLogins(SUBCLIENT1_LOGIN))
                        .withFieldNames(AgencyClientFieldEnum.LOGIN)
                        .withPage(new LimitOffsetMap().withOffset(-1L)),
                new Api5Error(4002, Api5ErrorDetails.OFFSET_MUST_BE_MORE_OR_EQUAL_THAN_0));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);

        assertThat("списалось правильное количество баллов", unitsBefore - unitsAfter,
                equalTo(CALL_COST + DEFAULT_ERROR_COST));
    }
}
