package ru.yandex.autotests.directapi.agencyclients.get;

import java.math.BigDecimal;
import java.util.Collections;

import com.yandex.direct.api.v5.agencyclients.AgencyClientFieldEnum;
import com.yandex.direct.api.v5.agencyclients.GetResponse;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.ClientsOptionsRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.UsersRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsFeatures;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsLogins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.agencyclients.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.ClientGetItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.math.BigDecimal.ROUND_HALF_UP;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

@Aqua.Test
@Features(AgencyClientsFeatures.GET)
@Description("Проверка получения VatRate")
public class GetVatRateTest {
    private static final String AGENCY_LOGIN = AgencyClientsLogins.AGENCY_RUB;
    private static final String SUBCLIENT_LOGIN = AgencyClientsLogins.AGENCY_RUB_SUBCLIENT;
    private static final String SUBCLIENT1_LOGIN = AgencyClientsLogins.AGENCY_RUB_SUBCLIENT1;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(AGENCY_LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Before
    @Step("Подгтовка тестовых данных")
    public void prepare() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AGENCY_LOGIN);
    }

    private BigDecimal convertDoubleToBigDecimal(double val) {
        return BigDecimal.valueOf(val).setScale(0, ROUND_HALF_UP);
    }

    @Test
    public void getSubclientVatRateTest() {
        double agencyVAT = 18;
        double clientVAT = 12;

        api.userSteps.clientFakeSteps().setVATRate(AGENCY_LOGIN, agencyVAT);

        // NB: выставим субклиенту НДС, отличный от агентского НДС, что бы убедиться что возвращается правильный (агентский) НДС
        api.userSteps.clientFakeSteps().setVATRate(SUBCLIENT_LOGIN, clientVAT);

        GetResponse getResponse = api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(SUBCLIENT_LOGIN)
                        .withFieldNames(AgencyClientFieldEnum.VAT_RATE));

        assertThat("полученное поле VATRate совпадает с ожидаемым", getResponse.getClients(),
                beanDifferV5(Collections.singletonList(
                        new ClientGetItemMap().withVatRate(convertDoubleToBigDecimal(agencyVAT))
                                .getBean())));
    }

    @Test
    public void getNotResidentSubclientVatRateTest() {
        double agencyVAT = 18;
        double clientVAT = 0;

        api.userSteps.clientFakeSteps().setVATRate(AGENCY_LOGIN, agencyVAT);

        // NB: выставим субклиенту-нерезиденту НДС, что бы убедиться что возвращается правильный (клиентский) НДС
        DirectJooqDbSteps directJooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(SUBCLIENT1_LOGIN);
        UsersRecord user = directJooqDbSteps.usersSteps().getUser(SUBCLIENT1_LOGIN);
        ClientsOptionsRecord clientsOptionsRecord = new ClientsOptionsRecord();
        clientsOptionsRecord.setClientid(user.getClientid());
        clientsOptionsRecord.setNonResident(1);
        directJooqDbSteps.clientsOptionsSteps().updateClientOptions(clientsOptionsRecord);

        api.userSteps.clientFakeSteps().setVATRate(SUBCLIENT1_LOGIN, clientVAT);

        GetResponse getResponse = api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(SUBCLIENT1_LOGIN)
                        .withFieldNames(AgencyClientFieldEnum.VAT_RATE));

        assertThat("полученное поле VATRate совпадает с ожидаемым", getResponse.getClients(),
                beanDifferV5(Collections.singletonList(
                        new ClientGetItemMap().withVatRate(convertDoubleToBigDecimal(clientVAT))
                                .getBean())));
    }
}
