package ru.yandex.autotests.directapi.agencyclients.get;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.agencyclients.AgencyClientFieldEnum;
import com.yandex.direct.api.v5.agencyclients.GetResponse;
import com.yandex.direct.api.v5.generalclients.ClientGetItem;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsFeatures;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsLogins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.agencyclients.AgencyClientsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.agencyclients.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Collections.emptyList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.both;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.not;

@Aqua.Test
@Features(AgencyClientsFeatures.GET)
@Description("Проверка видимости субклиентов для разных представителей")
@RunWith(Parameterized.class)
public class VisibilityTest {
    public static final String SUBCLIENT1 = AgencyClientsLogins.UPD_SUBCLIENT1;
    public static final String SUBCLIENT2 = AgencyClientsLogins.UPD_SUBCLIENT2;
    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter
    public String description;

    @Parameterized.Parameter(1)
    public String operatorLogin;

    @Parameterized.Parameter(2)
    public String agencyLogin;

    @Parameterized.Parameter(3)
    public List<Matcher<? super String>> expectedLogins;

    @Parameterized.Parameter(4)
    public List<Matcher<? super String>> unexpectedLogins;

    @Parameterized.Parameters(name = "{0} (operatorLogin = {1}, agencyLogin = {2})")
    public static Collection<Object[]> testData() {
        return Arrays.asList(new Object[][]{
                {"Главный представитель агентства (chief) видит всех",
                        AgencyClientsLogins.UPD_AGENCY,
                        null,
                        toMatcherCollection(SUBCLIENT1, SUBCLIENT2),
                        emptyList()},
                {"Основной представитель агентства (main) видит всех",
                        AgencyClientsLogins.UPD_AGENCY_UNLIM,
                        null,
                        toMatcherCollection(SUBCLIENT1, SUBCLIENT2),
                        emptyList()},
                {"Ограниченый представитель не видит всех субклиентов",
                        AgencyClientsLogins.UPD_AGENCY_LIM,
                        null,
                        toMatcherCollection(SUBCLIENT1),
                        toMatcherCollection(SUBCLIENT2)},
                {"Главный представитель агентства (chief) может видеть только субклиентов доступных ограниченному",
                        AgencyClientsLogins.UPD_AGENCY,
                        AgencyClientsLogins.UPD_AGENCY_LIM,
                        toMatcherCollection(SUBCLIENT1),
                        toMatcherCollection(SUBCLIENT2)}
        });
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void before() {
        api.as(operatorLogin);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(operatorLogin);
    }

    @Test
    public void rolesTest() {
        GetResponse getResponse = api.as(operatorLogin).userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap()
                        .withSelectionCriteria(new AgencyClientsSelectionCriteriaMap())
                        .withFieldNames(AgencyClientFieldEnum.LOGIN),
                agencyLogin);
        List<String> actualSubclients = getResponse.getClients().stream()
                .map(ClientGetItem::getLogin)
                .collect(Collectors.toList());
        assertThat(actualSubclients,
                both(containsInAnyOrder(expectedLogins))
                        .and(not(containsInAnyOrder(unexpectedLogins)))
        );
    }

    private static List<Matcher<? super String>> toMatcherCollection(String... logins) {
        return Arrays.stream(logins).map(Matchers::equalTo).collect(Collectors.toList());
    }
}
