package ru.yandex.autotests.directapi.agencyclients.update;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import com.yandex.direct.api.v5.agencyclients.AgencyClientFieldEnum;
import com.yandex.direct.api.v5.agencyclients.GetResponse;
import com.yandex.direct.api.v5.agencyclients.UpdateResponse;
import com.yandex.direct.api.v5.general.ClientsActionResult;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.generalclients.ClientGetItem;
import com.yandex.direct.api.v5.generalclients.GrantGetItem;
import org.hamcrest.Matcher;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.steps.UsersSteps;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsFeatures;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsLogins;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.agencyclients.AgencyClientUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.agencyclients.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.agencyclients.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ActionResultBaseMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.generalclients.ClientGetItemMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.ClientsActionResultMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.GrantGetItemMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.GrantItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static com.yandex.direct.api.v5.generalclients.PrivilegeEnum.EDIT_CAMPAIGNS;
import static com.yandex.direct.api.v5.generalclients.PrivilegeEnum.IMPORT_XLS;
import static com.yandex.direct.api.v5.generalclients.PrivilegeEnum.TRANSFER_MONEY;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.hasItems;
import static org.hamcrest.Matchers.hasProperty;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;


@Aqua.Test
@Features(AgencyClientsFeatures.UPDATE)
@Description("Проверка обновления Grants")
@RunWith(Parameterized.class)
public class UpdateGrantsTest {
    private static final String AGENCY_NAME = AgencyClientsLogins.UPD_AGENCY;
    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();
    @Parameterized.Parameter
    public String agencyLogin;

    @Parameterized.Parameter(1)
    public String subClientLogin;

    private Long subClientId;

    @Parameterized.Parameters(name = "agencyLogin {0} subClientLogin {1}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {AgencyClientsLogins.UPD_AGENCY, AgencyClientsLogins.UPD_SUBCLIENT1},
                {AgencyClientsLogins.UPD_AGENCY, AgencyClientsLogins.UPD_SUBCLIENT1_REP},
                {AgencyClientsLogins.UPD_AGENCY_LIM, AgencyClientsLogins.UPD_SUBCLIENT1},
                {AgencyClientsLogins.UPD_AGENCY_UNLIM, AgencyClientsLogins.UPD_SUBCLIENT1_REP},
        });
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void before() {
        subClientId = userStepsFor(subClientLogin).getUser(subClientLogin).getClientid();
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(agencyLogin);
    }

    @Test
    public void getGrantsAllOff() {
        api.as(agencyLogin);
        updateGrants(subClientId,
                new GrantItemMap(EDIT_CAMPAIGNS, YesNoEnum.NO),
                new GrantItemMap(IMPORT_XLS, YesNoEnum.NO),
                new GrantItemMap(TRANSFER_MONEY, YesNoEnum.NO));

        List<ClientGetItem> expectedResult = Collections.singletonList(
                (ClientGetItem) new ClientGetItemMap().withGrants(
                        new GrantGetItemMap(AGENCY_NAME, IMPORT_XLS, YesNoEnum.NO),
                        new GrantGetItemMap(AGENCY_NAME, EDIT_CAMPAIGNS, YesNoEnum.NO),
                        new GrantGetItemMap(AGENCY_NAME, TRANSFER_MONEY, YesNoEnum.NO)
                ).getBean()
        );

        GetResponse getResponse = api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(subClientLogin)
                        .withFieldNames(AgencyClientFieldEnum.GRANTS));

        List<ClientGetItem> actualResult = getResponse.getClients();

        actualResult.forEach(e -> e.getGrants().sort(Comparator.comparing(GrantGetItem::getPrivilege)));
        expectedResult.forEach(e -> e.getGrants().sort(Comparator.comparing(GrantGetItem::getPrivilege)));

        assertThat("полученные значения совпадают с ожидаемыми", actualResult, beanDifferV5(expectedResult));
    }

    @Test
    public void getGrantsAllOn() {
        api.as(agencyLogin);
        api.userSteps.agencyClientsStepsV5().agencyClientsUpdate(
                new UpdateRequestMap().withClients(
                        new AgencyClientUpdateItemMap()
                                .withClientId(subClientId)
                                .withGrants(
                                        new GrantItemMap(EDIT_CAMPAIGNS, YesNoEnum.YES),
                                        new GrantItemMap(IMPORT_XLS, YesNoEnum.YES),
                                        new GrantItemMap(TRANSFER_MONEY, YesNoEnum.YES)
                                )));

        List<ClientGetItem> expectedResult = Collections.singletonList(
                (ClientGetItem) new ClientGetItemMap().withGrants(
                        new GrantGetItemMap(AGENCY_NAME, IMPORT_XLS, YesNoEnum.YES),
                        new GrantGetItemMap(AGENCY_NAME, EDIT_CAMPAIGNS, YesNoEnum.YES),
                        new GrantGetItemMap(AGENCY_NAME, TRANSFER_MONEY, YesNoEnum.YES)
                ).getBean()
        );

        GetResponse getResponse = api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(subClientLogin)
                        .withFieldNames(AgencyClientFieldEnum.GRANTS));

        List<ClientGetItem> actualResult = getResponse.getClients();

        actualResult.forEach(e -> e.getGrants().sort(Comparator.comparing(GrantGetItem::getPrivilege)));
        expectedResult.forEach(e -> e.getGrants().sort(Comparator.comparing(GrantGetItem::getPrivilege)));

        assertThat("полученные значения совпадают с ожидаемыми", actualResult, beanDifferV5(expectedResult));
    }

    @Test
    public void repeatingSwitchingOffOneGrantShouldNotCauseAnError() {
        api.as(agencyLogin);
        GrantItemMap grantItemMap = new GrantItemMap(TRANSFER_MONEY, YesNoEnum.NO);
        updateGrants(subClientId, grantItemMap);
        UpdateResponse updateResponse = updateGrants(subClientId, grantItemMap);
        assertThat("результат не содержит ошибку",
                updateResponse.getUpdateResults(), contains(actionResultWithOutErrors()));
    }

    @Test
    public void repeatingSwitchingOnOneGrantShouldNotCauseAnError() {
        api.as(agencyLogin);
        GrantItemMap grantItemMap = new GrantItemMap(TRANSFER_MONEY, YesNoEnum.YES);
        updateGrants(subClientId, grantItemMap);
        UpdateResponse updateResponse = updateGrants(subClientId, grantItemMap);
        assertThat("результат не содержит ошибку",
                updateResponse.getUpdateResults(), contains(actionResultWithOutErrors()));
    }

    @Test
    public void switchingOffEditCampaignAndSwitchingOnImportXlsCauseAnError() {
        api.as(agencyLogin);
        UpdateResponse updateResponse = updateGrants(subClientId,
                new GrantItemMap(EDIT_CAMPAIGNS, YesNoEnum.NO),
                new GrantItemMap(IMPORT_XLS, YesNoEnum.YES));
        assertThat("результат содержит ожидаемую ошибку", updateResponse.getUpdateResults(), contains(
                actionResultWithError(
                        6000,
                        Api5ErrorDetailsJava.INCONSISTENT_STATE_ALLOW_EDIT_CAMPAIGN_AND_ALLOW_IMPORT_XLS,
                        path(ClientsActionResultMap.CLIENT_ID), subClientId
                )
        ));
    }

    @Test
    public void switchingOffEditCampaignCauseAnErrorWhenImportXlsSwitchedOn() {
        api.as(agencyLogin);
        UpdateResponse updateResponse = updateGrants(subClientId,
                new GrantItemMap(EDIT_CAMPAIGNS, YesNoEnum.YES),
                new GrantItemMap(IMPORT_XLS, YesNoEnum.YES));
        assumeThat("результат не содержит ошибку",
                updateResponse.getUpdateResults(), contains(actionResultWithOutErrors()));

        GetResponse getResponse = api.userSteps.agencyClientsStepsV5().agencyClientsGet(
                new GetRequestMap().withSubclientLogins(subClientLogin)
                        .withFieldNames(AgencyClientFieldEnum.GRANTS));

        List<ClientGetItem> actualResult = getResponse.getClients();
        Object importXlsOn = new GrantGetItemMap(AGENCY_NAME, IMPORT_XLS, YesNoEnum.YES).getBean();
        Object editCampaignOn = new GrantGetItemMap(AGENCY_NAME, EDIT_CAMPAIGNS, YesNoEnum.YES).getBean();
        assumeThat("EDIT_CAMPAIGNS и IMPORT_XLS включен",
                actualResult, contains(
                        hasProperty(ClientGetItemMap.GRANTS,
                                hasItems(beanDifferV5(editCampaignOn), beanDifferV5(importXlsOn)))
                )
        );

        updateResponse = updateGrants(subClientId, new GrantItemMap(EDIT_CAMPAIGNS, YesNoEnum.NO));
        assertThat("результат содержит ожидаемую ошибку", updateResponse.getUpdateResults(), contains(
                actionResultWithError(
                        6000,
                        Api5ErrorDetailsJava.INCONSISTENT_STATE_ALLOW_EDIT_CAMPAIGN_AND_ALLOW_IMPORT_XLS,
                        path(ClientsActionResultMap.CLIENT_ID), subClientId
                )
        ));
    }

    private UsersSteps userStepsFor(String login) {
        return api.userSteps.getDirectJooqDbSteps().useShardForLogin(login).usersSteps();
    }

    private UpdateResponse updateGrants(Long subClientId, GrantItemMap... grants) {
        return api.userSteps.agencyClientsStepsV5().agencyClientsUpdate(
                new UpdateRequestMap().withClients(
                        new AgencyClientUpdateItemMap()
                                .withClientId(subClientId)
                                .withGrants(grants)));
    }
    
    private Matcher<ClientsActionResult> actionResultWithOutErrors() {
        return hasProperty(ActionResultBaseMap.ERRORS, empty());
    }

    private Matcher<ClientsActionResult> actionResultWithError(int code, Api5ErrorDetailsJava details,
            Object... params)
    {
        return hasProperty(ActionResultBaseMap.ERRORS, contains(
                BeanDifferMatcher.beanDiffer(new Notification(code, details, params).getBean())));
    }
}
