package ru.yandex.autotests.directapi.agencyclients.update;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.UsersRecord;
import ru.yandex.autotests.direct.db.steps.UsersSteps;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsFeatures;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.agencyclients.AgencyClientUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.agencyclients.UpdateRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.CoreMatchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * https://st.yandex-team.ru/DIRECT-69110
 */
@Aqua.Test
@Features(AgencyClientsFeatures.UPDATE)
@Issue("https://st.yandex-team.ru/DIRECT-64387")
@Description("Проверка успешного обновления")
@RunWith(Parameterized.class)
public class UpdatePositiveTest {
    private static final String FIO_BEFORE = "Апдейт Клиентов";
    private static final String FIO_NEW = "Сакцес Апдейтов";

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter
    public String description;

    @Parameterized.Parameter(1)
    public String operatorLogin;

    @Parameterized.Parameter(2)
    public String agencyLogin;

    @Parameterized.Parameter(3)
    public String subClientLogin;

    private Long subClientId;

    private UsersSteps usersSteps;

    @Parameterized.Parameters(name = "{0}: operatorLogin {1} agencyLogin {2} subClientLogin {3}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {"Главный обновляет субклиента без указания Client-Login",
                        AgencyClientsLogins.UPD_AGENCY, null, AgencyClientsLogins.UPD_SUBCLIENT1},
                {"Главный обновляет субклиента",
                        AgencyClientsLogins.UPD_AGENCY, AgencyClientsLogins.UPD_AGENCY,
                        AgencyClientsLogins.UPD_SUBCLIENT1},
                {"Ограниченый обновляет доступного субклиента",
                        AgencyClientsLogins.UPD_AGENCY_LIM, null, AgencyClientsLogins.UPD_SUBCLIENT1},
                {"Неограниченный обновляет субклиента",
                        AgencyClientsLogins.UPD_AGENCY_UNLIM, null, AgencyClientsLogins.UPD_SUBCLIENT1},
                {"Ограниченый обновляет доступного субклиента (оператор -- главный)",
                        AgencyClientsLogins.UPD_AGENCY, AgencyClientsLogins.UPD_AGENCY_LIM,
                        AgencyClientsLogins.UPD_SUBCLIENT1},
                {"Неограниченный обновляет субклиента (оператор -- главный)",
                        AgencyClientsLogins.UPD_AGENCY, AgencyClientsLogins.UPD_AGENCY_UNLIM,
                        AgencyClientsLogins.UPD_SUBCLIENT1},
                {"Главный обновляет субклиента (оператор -- менедежер)",
                        AgencyClientsLogins.MANAGER_DEFAULT, AgencyClientsLogins.UPD_AGENCY,
                        AgencyClientsLogins.UPD_SUBCLIENT1},
                {"Ограниченый обновляет доступного субклиента (оператор -- менеджер)",
                        AgencyClientsLogins.MANAGER_DEFAULT, AgencyClientsLogins.UPD_AGENCY_LIM,
                        AgencyClientsLogins.UPD_SUBCLIENT1},
                {"Неограниченный обновляет другого субклиента (оператор -- менеджер)",
                        AgencyClientsLogins.MANAGER_DEFAULT, AgencyClientsLogins.UPD_AGENCY_UNLIM,
                        AgencyClientsLogins.UPD_SUBCLIENT1},
        });
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void prepare() {
        usersSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(subClientLogin).usersSteps();
        UsersRecord usersRecord = usersSteps.getUser(subClientLogin);
        usersRecord.setFio(FIO_BEFORE);
        usersSteps.updateUsers(usersRecord);
        subClientId = usersRecord.getClientid();
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(operatorLogin);
    }

    @Test
    public void successUpdateFioTest() {
        api.as(operatorLogin).userSteps.agencyClientsStepsV5().agencyClientsUpdate(
                new UpdateRequestMap().withClients(
                        new AgencyClientUpdateItemMap()
                                .withClientId(subClientId)
                                .withClientInfo(FIO_NEW)),
                agencyLogin
        );
        String fioAfterUpdate = usersSteps.getUser(subClientLogin).getFio();
        assertThat("имя пользователя изменилось на новое", fioAfterUpdate, equalTo(FIO_NEW));
    }
}
