package ru.yandex.autotests.directapi.agencyclients.update;

import com.yandex.direct.api.v5.agencyclients.UpdateResponse;
import com.yandex.direct.api.v5.general.ClientsActionResult;
import org.hamcrest.Matcher;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsFeatures;
import ru.yandex.autotests.directapi.agencyclients.AgencyClientsLogins;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.agencyclients.AgencyClientUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.agencyclients.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ActionResultBaseMap;
import ru.yandex.autotests.directapi.model.api5.general.ExceptionNotificationMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.ClientsActionResultMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.hasProperty;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(AgencyClientsFeatures.UPDATE)
@Description("Проверка списания баллов")
public class UpdateUnitsTest {
    private static final String AGENCY_LOGIN = AgencyClientsLogins.UPD_UNITS_AGENCY;
    private static final String SUBCLIENT1_LOGIN = AgencyClientsLogins.UPD_UNITS_SUBCLIENT1;
    private static final String SUBCLIENT2_LOGIN = AgencyClientsLogins.UPD_UNITS_SUBCLIENT2;
    private static final String OTHER_AGENCY_SUBCLIENT = AgencyClientsLogins.OTHER_AGENCY_SUBCLIENT;

    private static final int CALL_COST = 10;
    private static final int PER_OBJECT_COST = 1;
    private static final int COMMON_REQUEST_ERROR = 50;
    private static final int DEFAULT_ERROR_COST = 20;

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(AGENCY_LOGIN);
    public static final String INVALID_CLIENT_INFO = "";

    private int unitsBefore;

    @Before
    @Step("Подготовка тестовых данных")
    public void prepare() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AGENCY_LOGIN);
        unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);
    }

    @Test
    public void updateOne() {
        Long subClientId = getClientId(SUBCLIENT1_LOGIN);
        UpdateResponse response = api.userSteps.agencyClientsStepsV5().agencyClientsUpdate(
                new UpdateRequestMap().withClients(
                        new AgencyClientUpdateItemMap().withClientId(subClientId)));
        assumeThat("вернулся результат без ошибок", response.getUpdateResults(),
                contains(actionResultWithClientId(subClientId)));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);

        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter,
                equalTo(CALL_COST + PER_OBJECT_COST));
    }

    @Test
    public void updateNoClientsFoundTest() {
        Long subClientId = getClientId(OTHER_AGENCY_SUBCLIENT);
        UpdateResponse response = api.userSteps.agencyClientsStepsV5().agencyClientsUpdate(
                new UpdateRequestMap().withClients(
                        new AgencyClientUpdateItemMap().withClientId(subClientId)));

        // агентство не может обновлять данные чужого субклиента, должна вернуться ошибка "Not Found"
        assumeThat("вернулась ошибка", response.getUpdateResults(),
                contains(actionResultWithErrorCode(8800)));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);

        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter,
                equalTo(CALL_COST + DEFAULT_ERROR_COST));
    }

    @Test
    public void updateWithInvalidDataTest() {
        Long subClientId = getClientId(SUBCLIENT1_LOGIN);
        UpdateResponse response = api.userSteps.agencyClientsStepsV5().agencyClientsUpdate(
                new UpdateRequestMap().withClients(
                        new AgencyClientUpdateItemMap()
                                .withClientId(subClientId)
                                .withClientInfo(INVALID_CLIENT_INFO)
                ));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);

        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter,
                equalTo(CALL_COST + DEFAULT_ERROR_COST));
        // тестов на валидацию в direct-qa нет, поэтому тут на всякий случай assert
        assertThat("вернулась ошибка валидации", response.getUpdateResults(), contains(
                actionResultWithErrorCode(5003)));
    }

    @Test
    public void updateTwo() {
        Long subClient1Id = getClientId(SUBCLIENT1_LOGIN);
        Long subClient2Id = getClientId(SUBCLIENT2_LOGIN);
        UpdateResponse response = api.userSteps.agencyClientsStepsV5().agencyClientsUpdate(
                new UpdateRequestMap().withClients(
                        new AgencyClientUpdateItemMap().withClientId(subClient1Id),
                        new AgencyClientUpdateItemMap().withClientId(subClient2Id)
                )
        );
        assumeThat("вернулся результат без ошибок", response.getUpdateResults(),
                contains(
                        actionResultWithClientId(subClient1Id),
                        actionResultWithClientId(subClient2Id)
                )
        );

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);

        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter,
                equalTo(CALL_COST + 2 * PER_OBJECT_COST));
    }

    @Test
    public void updateValidAndInvalid() {
        Long subClient1Id = getClientId(SUBCLIENT1_LOGIN);
        Long subClient2Id = getClientId(SUBCLIENT2_LOGIN);
        UpdateResponse response = api.userSteps.agencyClientsStepsV5().agencyClientsUpdate(
                new UpdateRequestMap().withClients(
                        new AgencyClientUpdateItemMap().withClientId(subClient1Id),
                        new AgencyClientUpdateItemMap().withClientId(subClient2Id)
                                .withClientInfo(INVALID_CLIENT_INFO)
                )
        );
        assumeThat("вернулся результат без ошибок", response.getUpdateResults(),
                contains(
                        actionResultWithClientId(subClient1Id),
                        actionResultWithErrorCode(5003)
                )
        );

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);

        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter,
                equalTo(CALL_COST + PER_OBJECT_COST + DEFAULT_ERROR_COST));
    }

    @Test
    public void updateWithIncorrectCommonRequestTest() {
        api.userSteps.agencyClientsStepsV5().expectErrorOnAgencyClientsUpdate(
                new UpdateRequestMap(),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        capitalize(UpdateRequestMap.CLIENTS)));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(AGENCY_LOGIN);

        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(COMMON_REQUEST_ERROR));
    }

    private Long getClientId(String login) {
        return api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(login)
                .usersSteps()
                .getUser(login)
                .getClientid();
    }

    private Matcher<ClientsActionResult> actionResultWithErrorCode(int code) {
        return hasProperty(ActionResultBaseMap.ERRORS,
                contains(hasProperty(ExceptionNotificationMap.CODE, equalTo(code))));
    }

    private Matcher<ClientsActionResult> actionResultWithClientId(Long clientId) {
        return allOf(
                hasProperty(ClientsActionResultMap.CLIENT_ID, equalTo(clientId)),
                hasProperty(ActionResultBaseMap.ERRORS, empty())
        );
    }
}
