package ru.yandex.autotests.direct.api.audiencetargets.add;

import java.util.List;

import org.joda.time.DateTime;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.audiencetargets.AudienceTargetsFeatures;
import ru.yandex.autotests.direct.api.audiencetargets.AudienceTargetsLogins;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.GroupFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.AudienceTargetAddItemMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.retargeting.RetargetingGoalType;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by buhter on 03/03/16
 * https://st.yandex-team.ru/TESTIRT-8519
 */
@Aqua.Test
@Features(AudienceTargetsFeatures.ADD)
@Description("Общие проверки метода")
@Issue("https://st.yandex-team.ru/DIRECT-48447")
public class AddAudienceTargetsTest {
    private static final String LOGIN = AudienceTargetsLogins.LOGIN_TARGET9;
    private static Long pid1;
    private static Long pid2;
    private static Long listsId1;
    private static Long listsId2;
    private static Long cid;


    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);


    @BeforeClass
    public static void prepare() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        List<Long> goalIds = api.userSteps.retargetingSteps().getRetargetingGoalIDsByType(LOGIN, RetargetingGoalType.GOAL);
        assumeThat("у клиента должно быть > 1 цели", goalIds.size(), greaterThan(1));
        listsId1 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalIds.get(0));
        listsId2 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalIds.get(1));
    }

    @Before
    public void addRetargetingList() {
        pid1 = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        pid2 = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
    }

    @Test
    public void testSuccess() {
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(new AudienceTargetAddItemMap()
                                .withAdGroupId(pid1)
                                .withRetargetingListId(listsId1)
                                .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                        .getCurrency()).getLongMinPrice().longValue()))
                , ExpectedResult.success());
    }

    @Test
    public void checkAdGroupStatusBsSynced() {
        //DIRECT-58299
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid1);
        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(pid1, Status.YES);
        GroupFakeInfo groupFakeInfo = api.userSteps.groupFakeSteps().getGroupParams(pid1);
        assumeThat("группа синхронизирована с БК", groupFakeInfo.getStatusBsSynced(), equalTo(Status.YES));
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(new AudienceTargetAddItemMap()
                                .withAdGroupId(pid1)
                                .withRetargetingListId(listsId1)
                                .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                        .getCurrency()).getLongMinPrice().longValue()))
                , ExpectedResult.success());
        groupFakeInfo = api.userSteps.groupFakeSteps().getGroupParams(pid1);
        assertThat("группа не синхронизирована с БК", groupFakeInfo.getStatusBsSynced(), equalTo(Status.NO));
    }

    @Test
    public void checkAdGroupLastChange() {
        String lastChange = DateTime.now().minusMinutes(5).toString("yyyy-MM-dd hh:mm:ss");
        api.userSteps.groupFakeSteps().setLastChange(pid1, lastChange);
        GroupFakeInfo groupFakeInfo = api.userSteps.groupFakeSteps().getGroupParams(pid1);
        assumeThat("группе установился ожидаемый LastChange", groupFakeInfo.getLastChange(), equalTo(lastChange));
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(new AudienceTargetAddItemMap()
                                .withAdGroupId(pid1)
                                .withRetargetingListId(listsId1)
                                .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                        .getCurrency()).getLongMinPrice().longValue()))
                , ExpectedResult.success());
        groupFakeInfo = api.userSteps.groupFakeSteps().getGroupParams(pid1);
        assertThat("LastChange группы обновился", groupFakeInfo.getLastChange(), not(equalTo(lastChange)));
    }

    @Test
    public void testNoAdGroupId() {
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(new AudienceTargetAddItemMap()
                                .withRetargetingListId(listsId1)
                                .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                        .getCurrency()).getLongMinPrice().longValue()))
                , ExpectedResult.errors(new Notification(5005, Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                        capitalize(AudienceTargetAddItemMap.AD_GROUP_ID))));
    }

    @Test
    public void testSomeAdGroupIdWithDiffLists() {
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid1)
                                        .withRetargetingListId(listsId1)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue()),
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid2)
                                        .withRetargetingListId(listsId2)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue())),
                ExpectedResult.success(), ExpectedResult.success());
    }

    @Test
    public void testSomeAdGroupIdWithOneLists() {
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid1)
                                        .withRetargetingListId(listsId1)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue()),
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid2)
                                        .withRetargetingListId(listsId1)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue())),
                ExpectedResult.success(), ExpectedResult.success());
    }

    @Test
    public void testOneAdGroupIdWithSomeDiffLists() {
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid1)
                                        .withRetargetingListId(listsId1)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue()),
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid1)
                                        .withRetargetingListId(listsId2)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue())),
                ExpectedResult.success(), ExpectedResult.success());
    }

    @Test
    public void addSameItemsInOneRequest() {
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid1)
                                        .withRetargetingListId(listsId1)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue()),
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid1)
                                        .withRetargetingListId(listsId1)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue())),
                ExpectedResult.errors(new Notification(9800, Api5ErrorDetailsJava.RETARGETING_LISTS_UNIQUE)),
                ExpectedResult.errors(new Notification(9800, Api5ErrorDetailsJava.RETARGETING_LISTS_UNIQUE)));
    }

    @Test
    public void addSameItemsInTwoRequest() {
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid1)
                                        .withRetargetingListId(listsId1)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue())),
                ExpectedResult.success());
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid1)
                                        .withRetargetingListId(listsId1)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue())),
                ExpectedResult.errors(new Notification(9801, Api5ErrorDetailsJava.RETARGETING_LISTS_ALREADY_EXISTS_IN_GROUP)));
    }

    @Test
    public void oneValidAndOneInvalidTargets() {
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid1)
                                        .withRetargetingListId(listsId1)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue()),
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(0l)
                                        .withRetargetingListId(listsId2)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue())),
                ExpectedResult.success(),
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                        capitalize(AudienceTargetAddItemMap.AD_GROUP_ID))));
    }
}
