package ru.yandex.autotests.direct.api.audiencetargets.add;

import java.util.List;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.audiencetargets.AudienceTargetsFeatures;
import ru.yandex.autotests.direct.api.audiencetargets.AudienceTargetsLogins;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.AudienceTargetAddItemMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.retargeting.RetargetingGoalType;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * https://st.yandex-team.ru/TESTIRT-8519
 */
@Aqua.Test
@Features(AudienceTargetsFeatures.ADD)
@Description("Списание баллов API")
@Issue("https://st.yandex-team.ru/DIRECT-48447")
public class AddAudienceTargetsUnitsTest {
    private static final String LOGIN = AudienceTargetsLogins.LOGIN_TARGET10;
    private static final int ADD_COST = 10;
    private static final int ITEM_COST_ON_ADD = 2;
    private static final int GENERAL_FAULT_COST = 50;
    private static final int ERROR_COST = 20;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private Long pid1;
    private static Long listsId1;
    private static Long listsId2;
    private static Long cid;
    private Long pid2;

    @BeforeClass
    public static void prepare() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        List<Long> goalIds = api.userSteps.retargetingSteps().getRetargetingGoalIDsByType(LOGIN, RetargetingGoalType.GOAL);
        assumeThat("у клиента должно быть > 1 цели", goalIds.size(), greaterThan(1));
        listsId1 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalIds.get(0));
        listsId2 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalIds.get(1));
    }

    @Before
    public void init() {
        pid1 = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    public void addSuccess() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(new AudienceTargetAddItemMap()
                                .withAdGroupId(pid1)
                                .withRetargetingListId(listsId1)
                                .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                        .getCurrency()).getLongMinPrice().longValue())),
                ExpectedResult.success());
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assertThat("стоимость добавления одного объекта",
                unitsBefore - unitsAfter, equalTo(ADD_COST + ITEM_COST_ON_ADD));
    }

    @Test
    public void addSomeObject() {
        pid2 = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid1)
                                        .withRetargetingListId(listsId1)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue()),
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid2)
                                        .withRetargetingListId(listsId2)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue())),
                ExpectedResult.success(), ExpectedResult.success());
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assertThat("стоимость добавления двух объектов",
                unitsBefore - unitsAfter, equalTo(ADD_COST + ITEM_COST_ON_ADD * 2));
    }

    @Test
    public void addOneInvalidObject() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(-pid1)
                                        .withRetargetingListId(listsId1)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue())),
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                        capitalize(AudienceTargetAddItemMap.AD_GROUP_ID))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = ADD_COST + ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void addTwoInvalidRetargetingLists() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(
                                new AudienceTargetAddItemMap()
                                        .withRetargetingListId(listsId1)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue()),
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(-pid1)
                                        .withRetargetingListId(listsId2)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue())),
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                        capitalize(AudienceTargetAddItemMap.AD_GROUP_ID))),
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                        capitalize(AudienceTargetAddItemMap.AD_GROUP_ID))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = ADD_COST + ERROR_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void addOneValidAndOneInvalidRetargetingLists() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAudienceTargets(
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid1)
                                        .withRetargetingListId(listsId1)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue()),
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(-pid1)
                                        .withRetargetingListId(listsId2)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue())),
                ExpectedResult.success(),
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                        capitalize(AudienceTargetAddItemMap.AD_GROUP_ID))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = ADD_COST + ERROR_COST + ITEM_COST_ON_ADD;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void generalErrorOnAddRetargetingLists() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().expectErrorOnAdd(
                new AddRequestMap().withAudienceTargets(new AudienceTargetAddItemMap(null)),
                new Api5Error(8000, Api5ErrorDetailsJava.ARRAY_CONTAINS_NULL,
                        capitalize(AddRequestMap.AUDIENCE_TARGETS)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assertThat("списалось верное кол-во баллов", unitsBefore - unitsAfter, equalTo(GENERAL_FAULT_COST));
    }

    @Test
    public void notEnoughUnitsOnAddRetargetingLists() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(LOGIN, unitsBefore);
        unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assumeThat("доступные баллы пользователя", unitsBefore, equalTo(0));

        api.userSteps.audienceTargetsSteps().expectErrorOnAdd(new AddRequestMap()
                        .withAudienceTargets(new AudienceTargetAddItemMap()
                                .withAdGroupId(pid1)
                                .withRetargetingListId(listsId1)
                                .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                        .getCurrency()).getLongMinPrice().longValue())),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assertThat("доступные баллы пользователя после ошибки", unitsAfter, equalTo(0));
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }
}
