package ru.yandex.autotests.direct.api.audiencetargets.delete;

import java.util.List;

import org.json.simple.JSONObject;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.audiencetargets.AudienceTargetsFeatures;
import ru.yandex.autotests.direct.api.audiencetargets.AudienceTargetsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.apiclient.version5.ServiceNames;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.AudienceTargetAddItemMap;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.retargeting.RetargetingGoalType;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * https://st.yandex-team.ru/TESTIRT-8519
 */
@Aqua.Test
@Features(AudienceTargetsFeatures.DELETE)
@Description("Проверка списания баллов")
@Issue("https://st.yandex-team.ru/DIRECT-48447")
public class DeleteAudienceTargetsUnitsTest {
    private static final String LOGIN = AudienceTargetsLogins.LOGIN_TARGET19;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static final int CALL_COST = 10;
    private static final int AUDIENCE_TARGET_COST = 2;
    private static final int ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR_COST = 50;
    private Long targetId;
    private Long pid;
    private List<Long> goalIds;

    @Before
    public void addUnitsToClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        api.as(LOGIN);
        Long cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        pid = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        goalIds = api.userSteps.retargetingSteps().getRetargetingGoalIDsByType(LOGIN, RetargetingGoalType.GOAL);
        assumeThat("у клиента должно быть > 1 цели", goalIds.size(), greaterThan(1));
    }

    @Test
    public void deleteOneAudienceTarget() {
        Long retargetingListsId = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalIds.get(0));
        targetId = api.userSteps.audienceTargetsSteps().add(
                new AddRequestMap()
                        .withAudienceTargets(new AudienceTargetAddItemMap()
                                .withAdGroupId(pid)
                                .withRetargetingListId(retargetingListsId)
                        )).get(0);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().delete(targetId);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + AUDIENCE_TARGET_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void deleteTwoAudienceTarget() {
        Long retargetingListsId1 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalIds.get(0));
        Long retargetingListsId2 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalIds.get(1));
        List<Long> targetIds = api.userSteps.audienceTargetsSteps().add(
                new AddRequestMap()
                        .withAudienceTargets(
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid)
                                        .withRetargetingListId(retargetingListsId1),
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid)
                                        .withRetargetingListId(retargetingListsId2)
                        ));
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().delete(targetIds.get(0), targetIds.get(1));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + AUDIENCE_TARGET_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void deleteOneInvalidAudienceTarget() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        final long minusOne = -1L;
        api.userSteps.audienceTargetsSteps().shouldGetResultOnDelete(
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(minusOne)),
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetailsJava.ID_MUST_BE_POSITIVE_INTEGER)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void deleteTwoInvalidAudienceTarget() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        final long minusOne = -1L;
        api.userSteps.audienceTargetsSteps().shouldGetResultOnDelete(
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(minusOne, 1L)),
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetailsJava.ID_MUST_BE_POSITIVE_INTEGER)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AUDIENCE_TARGETS_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void deleteOneValidAndOneInvalidAudienceTarget() {
        Long retargetingListsId = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalIds.get(0));
        targetId = api.userSteps.audienceTargetsSteps().add(
                new AddRequestMap()
                        .withAudienceTargets(new AudienceTargetAddItemMap()
                                .withAdGroupId(pid)
                                .withRetargetingListId(retargetingListsId)
                        )).get(0);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        final long zeroId = 0L;
        api.userSteps.audienceTargetsSteps().shouldGetResultOnDelete(
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(targetId, zeroId)),
                ExpectedResult.success(targetId),
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetailsJava.ID_MUST_BE_POSITIVE_INTEGER)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST + AUDIENCE_TARGET_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void invalidRequest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        JSONObject request = new JSONObject();
        JSONObject selectionCriteria = new JSONObject();
        selectionCriteria.put(capitalize(IdsCriteriaMap.IDS), 1L);
        request.put(capitalize(DeleteRequestMap.SELECTION_CRITERIA), new Object[]{selectionCriteria});
        api.userSteps.retargetingListsSteps().shouldGetJSONErrorOn(
                ServiceNames.AUDIENCE_TARGETS,
                LOGIN,
                Action.DELETE,
                request,
                new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_NOT_CONTAIN_ARRAY,
                        capitalize(DeleteRequestMap.SELECTION_CRITERIA)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = COMMON_REQUEST_ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void deleteAudienceTargetWhenClientHasZeroUnits() {
        Long retargetingListsId = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalIds.get(0));
        targetId = api.userSteps.audienceTargetsSteps().add(
                new AddRequestMap()
                        .withAudienceTargets(new AudienceTargetAddItemMap()
                                .withAdGroupId(pid)
                                .withRetargetingListId(retargetingListsId)
                        )).get(0);
        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(LOGIN, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assumeThat("у клиента нету баллов", units, equalTo(0));

        api.userSteps.audienceTargetsSteps().expectErrorOnDelete(
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(targetId)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
    }

    private static String selectionCriteriaId(int index) {
        return path(DeleteRequestMap.SELECTION_CRITERIA, IdsCriteriaMap.IDS, index).toString();
    }
}
