package ru.yandex.autotests.direct.api.audiencetargets.setbids;

import java.util.Arrays;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.audiencetargets.AudienceTargetsFeatures;
import ru.yandex.autotests.direct.api.audiencetargets.AudienceTargetsLogins;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.AudienceTargetSetBidsItemMap;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.SetBidsRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.general.SetBidsExpectedResult;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.model.retargeting.RetargetingGoalType;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;

/**
 * https://st.yandex-team.ru/TESTIRT-8519
 */
@Aqua.Test
@Features(AudienceTargetsFeatures.SET_BIDS)
@Description("Проверка граничных значений ContextBid")
@Issue("https://st.yandex-team.ru/DIRECT-48447")
@RunWith(Parameterized.class)
public class SetBidsContextBidTresholdsTest {
    private Long targetId;
    private Money minimumBid;
    private Money maximumBid;

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public Currency currency;

    @Parameterized.Parameter(1)
    public String stringCurrency;

    @Parameterized.Parameter(2)
    public String login;

    @Parameterized.Parameters(name = "Currency = {0}")
    public static Iterable<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {Currency.RUB, MoneyCurrency.get(Currency.RUB).getJavaAPIAbbreviation(api.locale()),
                        AudienceTargetsLogins.LOGIN_RETARGETING_RUB},
                {Currency.UAH, MoneyCurrency.get(Currency.UAH).getJavaAPIAbbreviation(api.locale()),
                        AudienceTargetsLogins.LOGIN_RETARGETING_UAH},
                {Currency.KZT, MoneyCurrency.get(Currency.KZT).getJavaAPIAbbreviation(api.locale()),
                        AudienceTargetsLogins.LOGIN_RETARGETING_KZT},
                {Currency.EUR, MoneyCurrency.get(Currency.EUR).getJavaAPIAbbreviation(api.locale()),
                        AudienceTargetsLogins.LOGIN_RETARGETING_EUR},
                {Currency.USD, MoneyCurrency.get(Currency.USD).getJavaAPIAbbreviation(api.locale()),
                        AudienceTargetsLogins.LOGIN_RETARGETING_USD},
                {Currency.CHF, MoneyCurrency.get(Currency.CHF).getJavaAPIAbbreviation(api.locale()),
                        AudienceTargetsLogins.LOGIN_RETARGETING_CHF},
                {Currency.TRY, MoneyCurrency.get(Currency.TRY).getJavaAPIAbbreviation(api.locale()),
                        AudienceTargetsLogins.LOGIN_RETARGETING_TRY}
        });
    }

    private long adGroupId;

    @Before
    @Step("Подготовка данных для теста")
    public void createAdGroup() {
        api.as(login);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(login);
        long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        Long pid = api.userSteps.adGroupsSteps().addDefaultGroup((int) campaignId);
        Long goalId = api.userSteps.retargetingSteps().getRetargetingGoalIDsByType(login, RetargetingGoalType.GOAL).get(0);
        Long retargetingListsId = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalId);
        targetId = api.userSteps.audienceTargetsSteps().addWithRetargetingList(pid,retargetingListsId);
        minimumBid = MoneyCurrency.get(currency).getMinPrice();
        maximumBid = MoneyCurrency.get(currency).getMaxPrice();
    }

    @Test
    public void setMinimumContextBid() {
        api.userSteps.audienceTargetsSteps().shouldGetResultOnSetBids(
                new SetBidsRequestMap().withBids(
                        new AudienceTargetSetBidsItemMap()
                                .withId(targetId)
                                .withContextBid(minimumBid.bidLong().longValue())
                ), SetBidsExpectedResult.successWithId());
    }

    @Test
    public void setLessThanMinimumContextBid() {
        String minimumBidInError = MoneyCurrency.get(currency).getMinPrice().stringValue(MoneyFormat.TWO_DIGITS_POINT);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnSetBids(
                new SetBidsRequestMap().withBids(new AudienceTargetSetBidsItemMap()
                        .withId(targetId)
                        .withContextBid(minimumBid.getPrevious().bidLong().longValue())),
                SetBidsExpectedResult.errors(new Notification(5005, Api5ErrorDetailsJava.VALUE_MUST_NOT_BE_LESS,
                        capitalize(KeywordAddMap.CONTEXT_BID), minimumBidInError, stringCurrency)));
    }

    @Test
    public void setMaximumContextBid() {
        api.userSteps.audienceTargetsSteps().shouldGetResultOnSetBids(
                new SetBidsRequestMap().withBids(
                        new AudienceTargetSetBidsItemMap()
                                .withId(targetId)
                                .withContextBid(maximumBid.bidLong().longValue())
                ), SetBidsExpectedResult.successWithId());
    }

    @Test
    public void setGreaterThanMaximumContextBid() {
        String maximumBidInError = MoneyCurrency.get(currency).getMaxPrice().stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnSetBids(
                new SetBidsRequestMap().withBids(new AudienceTargetSetBidsItemMap()
                        .withId(targetId)
                        .withContextBid(maximumBid.getNext().bidLong().longValue())),
                SetBidsExpectedResult.errors(new Notification(5005, Api5ErrorDetailsJava.VALUE_MUST_NOT_BE_MORE,
                        capitalize(KeywordAddMap.CONTEXT_BID), maximumBidInError, stringCurrency)));
    }
}
