package ru.yandex.autotests.direct.api.audiencetargets.setbids;

import java.util.List;

import com.yandex.direct.api.v5.general.PriorityEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.audiencetargets.AudienceTargetsFeatures;
import ru.yandex.autotests.direct.api.audiencetargets.AudienceTargetsLogins;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.AudienceTargetAddItemMap;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.AudienceTargetGetItemMap;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.AudienceTargetSetBidsItemMap;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.SetBidsRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.general.SetBidsExpectedResult;
import ru.yandex.autotests.directapi.model.retargeting.RetargetingGoalType;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * https://st.yandex-team.ru/TESTIRT-8519
 */
@Aqua.Test
@Features(AudienceTargetsFeatures.SET_BIDS)
@Description("Проверка списания баллов")
@Issue("https://st.yandex-team.ru/DIRECT-48447")
public class SetBidsUnitsTest {
    private static final String LOGIN = AudienceTargetsLogins.LOGIN_TARGET57;

    private static final long INITIAL_CONTEXT_PRICE =
            MoneyCurrency.get(Currency.RUB).getMinPrice().bidLong().longValue();
    private static final long NEW_CONTEXT_PRICE =
            MoneyCurrency.get(Currency.RUB).getMinPrice().getNext().bidLong().longValue();

    private static final int CALL_COST = 10;
    private static final int RETARGETING_LIST_COST = 2;
    private static final int ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR_COST = 50;

    private static Long cid;
    private static Long pid;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);
    private static List<Long> targetIds;

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void addUnitsToClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        api.as(LOGIN);
        cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        pid = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        List<Long> goalIds =
                api.userSteps.retargetingSteps().getRetargetingGoalIDsByType(LOGIN, RetargetingGoalType.GOAL);
        assumeThat("у клиента должно быть > 1 цели", goalIds.size(), greaterThan(1));
        Long retargetingListsId1 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalIds.get(0));
        Long retargetingListsId2 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalIds.get(1));
        targetIds = api.userSteps.audienceTargetsSteps().add(
                new AddRequestMap()
                        .withAudienceTargets(
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid)
                                        .withRetargetingListId(retargetingListsId1),
                                new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid)
                                        .withRetargetingListId(retargetingListsId2)
                        ));
    }

    @Before
    public void initAndClearUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        api.userSteps.audienceTargetsSteps().setBids(
                new SetBidsRequestMap().withBids(
                        new AudienceTargetSetBidsItemMap()
                                .withCampaignId(cid)
                                .withContextBid(INITIAL_CONTEXT_PRICE)
                ));
    }

    @Test
    public void setBidsOneAudienceTarget() {
        api.userSteps.audienceTargetsSteps().suspend(targetIds.get(0));
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap()
                        .withBids(new AudienceTargetSetBidsItemMap()
                                .withId(targetIds.get(0))
                                .withContextBid(NEW_CONTEXT_PRICE)),
                SetBidsExpectedResult.successWithId());
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + RETARGETING_LIST_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void setBidsTwoAudienceTarget() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap()
                        .withBids(
                                new AudienceTargetSetBidsItemMap()
                                        .withId(targetIds.get(0))
                                        .withContextBid(NEW_CONTEXT_PRICE),
                                new AudienceTargetSetBidsItemMap()
                                        .withId(targetIds.get(1))
                                        .withContextBid(NEW_CONTEXT_PRICE)),
                SetBidsExpectedResult.successWithId(), SetBidsExpectedResult.successWithId());
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + RETARGETING_LIST_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void withdrawUnitsForCallOnlyWhenContextPriceDoesNotChange() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap()
                        .withBids(
                                new AudienceTargetSetBidsItemMap()
                                        .withId(targetIds.get(0))
                                        .withContextBid(INITIAL_CONTEXT_PRICE),
                                new AudienceTargetSetBidsItemMap()
                                        .withId(targetIds.get(1))
                                        .withContextBid(INITIAL_CONTEXT_PRICE)),
                SetBidsExpectedResult.successWithId(), SetBidsExpectedResult.successWithId());
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        assertThat("списалось верное число баллов", actualCost, equalTo(CALL_COST));
    }

    @Test
    public void setBidsTwoAudienceTargetByAdGroupId() {
        //DIRECT-58441
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap()
                        .withBids(
                                new AudienceTargetSetBidsItemMap()
                                        .withAdGroupId(pid)
                                        .withContextBid(NEW_CONTEXT_PRICE)),
                SetBidsExpectedResult.successWithAdGroupId());
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + RETARGETING_LIST_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void setBidsTwoAudienceTargetByCampaignId() {
        //DIRECT-58441
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap()
                        .withBids(
                                new AudienceTargetSetBidsItemMap()
                                        .withCampaignId(cid)
                                        .withContextBid(NEW_CONTEXT_PRICE)),
                SetBidsExpectedResult.successWithCampaignId());
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + RETARGETING_LIST_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void setBidsOneInvalidAudienceTarget() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap()
                        .withBids(
                                new AudienceTargetSetBidsItemMap()
                                        .withId(1L)
                                        .withStrategyPriority(PriorityEnum.LOW)),
                SetBidsExpectedResult.errors(
                        new Notification(8800, Api5ErrorDetailsJava.AUDIENCE_TARGETS_FOR_PARAMETERS_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void setBidsTwoInvalidAudienceTarget() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap()
                        .withBids(
                                new AudienceTargetSetBidsItemMap()
                                        .withId(1L)
                                        .withStrategyPriority(PriorityEnum.HIGH),
                                new AudienceTargetSetBidsItemMap()
                                        .withId(-1L)
                                        .withStrategyPriority(PriorityEnum.LOW)),
                SetBidsExpectedResult.errors(
                        new Notification(8800, Api5ErrorDetailsJava.AUDIENCE_TARGETS_FOR_PARAMETERS_NOT_FOUND)),
                SetBidsExpectedResult.errors(
                        new Notification(5005, Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                                path(AudienceTargetSetBidsItemMap.ID))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void setBidsOneValidAndOneInvalidAudienceTarget() {
        api.userSteps.audienceTargetsSteps().suspend(targetIds.get(0));
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.audienceTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap()
                        .withBids(
                                new AudienceTargetSetBidsItemMap()
                                        .withId(targetIds.get(0))
                                        .withContextBid(NEW_CONTEXT_PRICE),
                                new AudienceTargetSetBidsItemMap()
                                        .withId(-1L)
                                        .withStrategyPriority(PriorityEnum.LOW)),
                SetBidsExpectedResult.successWithId(),
                SetBidsExpectedResult.errors(
                        new Notification(5005, Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                                path(AudienceTargetGetItemMap.ID))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST + RETARGETING_LIST_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void invalidRequest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);

        api.userSteps.audienceTargetsSteps().expectErrorOnSetBids(new SetBidsRequestMap(),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        path(SetBidsRequestMap.BIDS)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        assertThat("списалось верное число баллов", actualCost, equalTo(COMMON_REQUEST_ERROR_COST));
    }

    @Test
    public void setBidsAudienceTargetWhenClientHasZeroUnits() {
        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(LOGIN, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assumeThat("у клиента нету баллов", units, equalTo(0));

        api.userSteps.audienceTargetsSteps().expectErrorOnSetBids(
                new SetBidsRequestMap()
                        .withBids(
                                new AudienceTargetSetBidsItemMap()
                                        .withId(targetIds.get(0))
                                        .withStrategyPriority(PriorityEnum.HIGH)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
    }
}
