package ru.yandex.autotests.direct.api.bidmodifiers.add;

import java.util.Arrays;
import java.util.Collection;
import java.util.function.Supplier;

import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.CampaignFakeInfo;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.GroupFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.DemographicsAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.MobileAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.RegionalAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.RetargetingAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.SerpLayoutAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.VideoAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.general.MultiIdsExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 21.07.2015.
 */
@Aqua.Test
@Features(BidModifiersFeatures.ADD)
@Description("Проверка у кампании времени последнего изменения и статуса синхронизации с БК при добавлении " +
        "корректировки ставок")
@RunWith(Parameterized.class)
public class AddBidModifiersBsSyncedCampaignTest {
    private static LogSteps log = LogSteps.getLogger(AddBidModifiersBsSyncedCampaignTest.class);
    private static final String client = BidModifiersLogins.SINGLE_CLIENT_ADD;

    private static DateTime lastChange = DateTime.now().minusDays(5);
    private static final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");

    private Long campaignID;
    private Long groupID;
    private static Integer conditionID;
    private Long bannerID;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public Supplier<BidModifierAddMap> bidModifierAddMapGetter;

    @Parameterized.Parameters(name = "{0}")
    public static Collection testData() {
        Object[][] data = new Object[][]{
                {"withMobileAdjustment", (Supplier<BidModifierAddMap>)
                        () -> new BidModifierAddMap().withMobileAdjustment(
                                new MobileAdjustmentMap().defaultMobileAdjustment())},
                {"withDemographicsAdjustment", (Supplier<BidModifierAddMap>)
                        () -> new BidModifierAddMap().withDemographicsAdjustment(
                                new DemographicsAdjustmentMap().defaultDemographicsAdjustment())},
                {"withRetargetingAdjustment", (Supplier<BidModifierAddMap>)
                        () -> new BidModifierAddMap().withRetargetingAdjustment(
                                new RetargetingAdjustmentMap().defaultRetargetingAdjustment()
                                        .withRetargetingConditionId(conditionID))},
                {"withRegionalAdjustment", (Supplier<BidModifierAddMap>)
                        () -> new BidModifierAddMap().withRegionalAdjustment(
                                new RegionalAdjustmentMap().defaultRegionalAdjustment())},
                {"withVideoAdjustment", (Supplier<BidModifierAddMap>)
                        () -> new BidModifierAddMap().withVideoAdjustment(
                                new VideoAdjustmentMap().defaultVideoAdjustment())},
                {"withSerpLayoutAdjustment", (Supplier<BidModifierAddMap>)
                        () -> new BidModifierAddMap().withSerpLayoutAdjustment(
                                new SerpLayoutAdjustmentMap().defaultSerpLayoutAdjustment())},
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void prepareGroup() {
        //DIRECT-44185
        conditionID = api.userSteps.retargetingSteps().addRandomRetargetingCondition(client);
    }

    @Before
    public void before() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        bannerID = api.userSteps.adsSteps().addDefaultTextAd(groupID);
        log.info("Промодерируем фейково");
        api.userSteps.campaignFakeSteps().makeCampaignModerated(campaignID);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(groupID);
        api.userSteps.bannersFakeSteps().makeBannersModerated(bannerID);

        log.info("Выставим статусы синхронизации с БК фейково");
        api.userSteps.campaignFakeSteps().setBSSynced(campaignID, true);
        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(groupID, Status.YES);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(bannerID, Status.YES);

        log.info("Сдвинем время последнего изменения на несколько дней");
        api.userSteps.campaignFakeSteps().setLastChange(campaignID, lastChange.toString());
        api.userSteps.groupFakeSteps().setLastChange(groupID, lastChange.toString());
        api.userSteps.bannersFakeSteps().setLastChange(bannerID, lastChange.toString());

        log.info("Добавим корректировку ставок");
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(bidModifierAddMapGetter.get()
                        .withCampaignId(campaignID)),
                MultiIdsExpectedResult.success());
    }

    @Test
    public void campaignStatusBsSyncedTest() {
        CampaignFakeInfo campaignInfo = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(campaignID);

        assertThat("статус синхронизации с БК у кампании после добавления корректировки ставок",
                campaignInfo.getStatusBsSynced(), equalTo(Status.NO));
    }

    @Test
    public void groupStatusBsSyncedTest() {
        GroupFakeInfo groupInfo = api.userSteps.groupFakeSteps().getGroupParams(groupID);

        assertThat("статус синхронизации с БК у группы после добавления корректировки ставок",
                groupInfo.getStatusBsSynced(), equalTo(Status.YES));
    }

    @Test
    public void campaignLastChangeTest() {
        CampaignFakeInfo campaignInfo = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(campaignID);
        assumeThat("время последнего изменения кампании", campaignInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(campaignInfo.getLastChange());

        assertThat("время последнего изменения кампании после добавления корректировки ставок", date,
                equalTo(lastChange.toLocalDate()));
    }

    @Test
    public void groupLastChangeTest() {
        GroupFakeInfo groupInfo = api.userSteps.groupFakeSteps().getGroupParams(groupID);
        assumeThat("время последнего изменения группы", groupInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(groupInfo.getLastChange());

        assertThat("время последнего изменения группы после добавления корректировки ставок", date,
                equalTo(lastChange.toLocalDate()));
    }
}
