package ru.yandex.autotests.direct.api.bidmodifiers.add;

import java.util.List;

import com.yandex.direct.api.v5.general.AgeRangeEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.DemographicsAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.MobileAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.RegionalAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.RetargetingAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.general.MultiIdsExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;

/**
 * Created by chicos on 16.07.2015.
 */
@Aqua.Test
@Features(BidModifiersFeatures.ADD)
@Description("Негативные сценарии добавления множественных корректировок ставок")
public class AddBidModifiersMultipleNegativeTest {
    private static final String client = BidModifiersLogins.SINGLE_CLIENT_ADD;

    private static Long campaignID;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Before
    public void prepareCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
    }

    @Test
    @Description("Попытка добавления различных типов корректировок в одном объекте BidModifier")
    public void allAdjustmentTypesInOneBidModifierTest() {
        Integer conditionId = api.userSteps.retargetingSteps().addRandomRetargetingCondition(client);
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(
                        new BidModifierAddMap()
                                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                        .defaultDemographicsAdjustment())
                                .withRetargetingAdjustment(new RetargetingAdjustmentMap()
                                        .defaultRetargetingAdjustment()
                                        .withRetargetingConditionId(conditionId))
                                .withRegionalAdjustment(new RegionalAdjustmentMap()
                                        .defaultRegionalAdjustment())
                        .withCampaignId(campaignID)),
                MultiIdsExpectedResult.errors(new Notification(5009,
                        Api5ErrorDetails.ONLY_ONE_OF_FIELDS_SHOULD_BE_SPECIFIED_BM,
                        BidModifierAddMap.getBidModifiersMandatoryAdjustmentNames())));
    }

    @Test
    @Description("При добавлении набора демографических корректировок ставок возникающая ошибка - общая для всего набора")
    public void multipleDemographicsAdjustmentSingleErrorTest() {
        Long groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);

        MultiIdsExpectedResult perlExpectedResult = MultiIdsExpectedResult.errors(new Notification(5005,
                Api5ErrorDetails.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER));
        MultiIdsExpectedResult javaExpectedResult = MultiIdsExpectedResult.errors(new Notification(5005,
                Api5ErrorDetailsJava.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER));

        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(
                        new BidModifierAddMap()
                                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                                .defaultDemographicsAdjustment(),
                                        new DemographicsAdjustmentMap()
                                                .withAge(AgeRangeEnum.AGE_25_34)
                                                .withBidModifier(DemographicsAdjustmentMap.DEMOGRAPHICS_ADJUSTMENT_MIN - 1))
                                .withAdGroupId(groupID)),
                singletonList(perlExpectedResult), singletonList(javaExpectedResult));
    }

    @Test
    @Description("При добавлении набора корректировок ставок ретаргетинга возникающая ошибка - общая для всего набора")
    public void multipleRetargetingAdjustmentSingleErrorTest() {
        Long groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        Integer conditionId1 = api.userSteps.retargetingSteps().addRandomRetargetingCondition(client);
        Integer conditionId2 = api.userSteps.retargetingSteps().addRandomRetargetingCondition(client);

        MultiIdsExpectedResult perlExpectedResult = MultiIdsExpectedResult.errors(new Notification(5005,
                Api5ErrorDetails.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER));
        MultiIdsExpectedResult javaExpectedResult = MultiIdsExpectedResult.errors(new Notification(5005,
                Api5ErrorDetailsJava.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER));

        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(
                        new BidModifierAddMap()
                                .withRetargetingAdjustment(new RetargetingAdjustmentMap()
                                                .defaultRetargetingAdjustment()
                                                .withRetargetingConditionId(conditionId1),
                                        new RetargetingAdjustmentMap()
                                                .withRetargetingConditionId(conditionId2)
                                                .withBidModifier(RetargetingAdjustmentMap.RETARGETING_ADJUSTMENT_MIN - 1))
                                .withAdGroupId(groupID)
                ), singletonList(perlExpectedResult), singletonList(javaExpectedResult));
    }

    @Test
    @Description("При добавлении нескольких корректировок ставок одного типа, одна из которх некорректна, " +
            "результат операции выводится для каждого объекта BidModifier раздельно")
    public void multipleModifiersOneInvalidSameTypesTest() {
        Long campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign(client);

        MultiIdsExpectedResult perlExpectedResult = MultiIdsExpectedResult.errors(new Notification(5005,
                Api5ErrorDetails.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER));
        MultiIdsExpectedResult javaExpectedResult = MultiIdsExpectedResult.errors(new Notification(5005,
                Api5ErrorDetailsJava.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER));

        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(
                        new BidModifierAddMap()
                                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                        .defaultDemographicsAdjustment())
                                .withCampaignId(campaignID),
                        new BidModifierAddMap()
                                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                        .withAge(AgeRangeEnum.AGE_25_34)
                                        .withBidModifier(DemographicsAdjustmentMap.DEMOGRAPHICS_ADJUSTMENT_MIN - 1))
                                .withCampaignId(campaignID2)),
                asList(MultiIdsExpectedResult.success(), perlExpectedResult),
                asList(MultiIdsExpectedResult.success(), javaExpectedResult));
    }

    @Test
    @Description("При добавлении нескольких некорректных корректировок ставок одного типа " +
            "результат операции выводится для каждого объекта BidModifier раздельно")
    public void multipleModifiersAllInvalidSameTypesTest() {
        Long campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign(client);

        MultiIdsExpectedResult commonFirstError = MultiIdsExpectedResult.errors(new Notification(5005,
                Api5ErrorDetails.BID_MODIFIER_SHOULD_BE_NOT_MORE,
                DemographicsAdjustmentMap.DEMOGRAPHICS_ADJUSTMENT_MAX));

        List<MultiIdsExpectedResult> perlExpectedResult = asList(
                commonFirstError,
                MultiIdsExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetails.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER)));
        List<MultiIdsExpectedResult> javaExpectedResult = asList(
                commonFirstError,
                MultiIdsExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetailsJava.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER)));

        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(
                        new BidModifierAddMap()
                                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                        .withAge(AgeRangeEnum.AGE_0_17)
                                        .withBidModifier(DemographicsAdjustmentMap.DEMOGRAPHICS_ADJUSTMENT_MAX + 1))
                                .withCampaignId(campaignID),
                        new BidModifierAddMap()
                                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                        .withAge(AgeRangeEnum.AGE_25_34)
                                        .withBidModifier(DemographicsAdjustmentMap.DEMOGRAPHICS_ADJUSTMENT_MIN - 1))
                                .withCampaignId(campaignID2)),
                perlExpectedResult, javaExpectedResult);
    }

    @Test
    @Description("При добавлении нескольких корректировок ставок разных типов, одна из которых некорректна, " +
            "результат операции выводится для каждого объекта BidModifier раздельно")
    public void multipleModifiersOneInvalidDifferentTypesTest() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(
                        new BidModifierAddMap()
                                .withMobileAdjustment(new MobileAdjustmentMap()
                                        .withBidModifier(-1))
                                .withCampaignId(campaignID),
                        new BidModifierAddMap()
                                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                        .defaultDemographicsAdjustment())
                                .withCampaignId(campaignID)),
                MultiIdsExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetails.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER)),
                MultiIdsExpectedResult.success());
    }

}
