package ru.yandex.autotests.direct.api.bidmodifiers.add;

import java.util.Arrays;

import com.yandex.direct.api.v5.general.AgeRangeEnum;
import com.yandex.direct.api.v5.general.GenderEnum;
import org.apache.commons.lang3.StringUtils;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.utils.beans.JsonObject;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.DemographicsAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.MobileAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.general.MultiIdsExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

/**
 * Created by chicos on 21.07.2015.
 */
@Aqua.Test
@Features(BidModifiersFeatures.ADD)
@Description("Проверка некорректной структуры запроса Add")
public class AddBidModifiersStructureNegativeTest {
    private static final String client = BidModifiersLogins.SINGLE_CLIENT_ADD;

    private static Long groupID;
    private static Long campaignID;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client).protocol(ProtocolType.JSON);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepareGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
    }

    @Test
    public void invalidActionNameTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.ACTION_INVALID,
                new AddRequestMap().withBidModifiers(new BidModifierAddMap()
                        .defaultBidModifier()
                        .withAdGroupId(groupID)),
                new Api5Error(55, Api5ErrorDetailsJava.INVALID_METHOD_VALUE));
    }

    @Test
    public void emptyActionNameTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.ACTION_EMPTY,
                new AddRequestMap().withBidModifiers(new BidModifierAddMap()
                        .defaultBidModifier()
                        .withAdGroupId(groupID)),
                new Api5Error(55, Api5ErrorDetailsJava.INVALID_METHOD_VALUE));
    }

    @Test
    public void noBidModifiersElementInRequestTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.ADD,
                new AddRequestMap(),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        "params",
                        StringUtils.capitalize(AddRequestMap.BID_MODIFIERS)),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        StringUtils.capitalize(AddRequestMap.BID_MODIFIERS)));
    }

    @Test
    public void emptyBidModifiersPackTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(null),
                new Api5Error(8000,
                        Api5ErrorDetails.FIELD_MUST_CONTAIN_NOT_LESS_ELEMENTS,
                        StringUtils.capitalize(AddRequestMap.BID_MODIFIERS)),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_NOT_LESS_ELEMENTS,
                        StringUtils.capitalize(AddRequestMap.BID_MODIFIERS), 1));
    }

    @Test
    public void integerArrayInBidModifiersTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorJsonOn(
                Action.ADD,
                new JsonObject()
                        .with(StringUtils.capitalize(AddRequestMap.BID_MODIFIERS), Arrays.asList(123, 456, 789)),
                new Api5Error(8000, Api5ErrorDetails.ELEMENT_OF_ARRAY_MUST_CONTAIN_OBJECT,
                        StringUtils.capitalize(AddRequestMap.BID_MODIFIERS)),
                new Api5Error(8000, Api5ErrorDetailsJava.INVALID_VALUE,
                        StringUtils.capitalize(AddRequestMap.BID_MODIFIERS)));
    }

    @Test
    public void emptyAdjustmentPackTest() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(new BidModifierAddMap()
                        .withDemographicsAdjustment()
                        .withAdGroupId(groupID)),
                MultiIdsExpectedResult.errors(new Notification(5008,
                        Api5ErrorDetails.ONE_OF_FIELDS_SHOULD_BE_SPECIFIED_BM,
                        BidModifierAddMap.getBidModifiersMandatoryAdjustmentNames())));
    }

    @Test
    @Description("Попытка добавления BidModifier без корректировок")
    public void noAdjustmentsInRequestTest() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(
                        new BidModifierAddMap()
                                .withAdGroupId(groupID)),
                MultiIdsExpectedResult.errors(new Notification(5008,
                        Api5ErrorDetails.ONE_OF_FIELDS_SHOULD_BE_SPECIFIED_BM,
                        BidModifierAddMap.getBidModifiersMandatoryAdjustmentNames())));
    }

    @Test
    @Description("Попытка добавления одной корректировки одновременно в группу и кампанию")
    public void addToCampaignAndGroupTest() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(
                        new BidModifierAddMap()
                                .defaultBidModifier()
                                .withAdGroupId(groupID)
                                .withCampaignId(campaignID)),
                MultiIdsExpectedResult.errors(new Notification(5009,
                        Api5ErrorDetails.ONLY_ONE_OF_FIELDS_SHOULD_BE_SPECIFIED_BM,
                        BidModifierAddMap.getBidModifiersMandatoryIdNames())));
    }

    @Test
    @Description("Попытка добавления корректировки без указания группы или кампании")
    public void noCampaignAndGroupIdTest() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(
                        new BidModifierAddMap().defaultBidModifier()),
                MultiIdsExpectedResult.errors(new Notification(5008,
                        Api5ErrorDetails.ONE_OF_FIELDS_SHOULD_BE_SPECIFIED_BM,
                        BidModifierAddMap.getBidModifiersMandatoryIdNames())));
    }

    @Test
    @Description("Попытка добавления двух однотипных корректировок ставок в одну группу")
    public void addModifierWithSameAdjustmentTypesInRequestTest() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(
                        new BidModifierAddMap()
                                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                        .withDefaultBidModifier()
                                        .withAge(AgeRangeEnum.AGE_0_17))
                                .withAdGroupId(groupID),
                        new BidModifierAddMap()
                                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                        .withDefaultBidModifier()
                                        .withGender(GenderEnum.GENDER_FEMALE))
                                .withAdGroupId(groupID)),
                MultiIdsExpectedResult
                        .errors(new Notification(9800, Api5ErrorDetails.BID_MODIFIER_SHOULD_NOT_BE_PRESENTED_TWICE)),
                MultiIdsExpectedResult
                        .errors(new Notification(9800, Api5ErrorDetails.BID_MODIFIER_SHOULD_NOT_BE_PRESENTED_TWICE)));
    }

    @Test
    public void unknownGenderEnumValueTest() {
        String genderValue = "UnknownEnumValue";

        JsonObject addRequest = new JsonObject()
                .with(StringUtils.capitalize(AddRequestMap.BID_MODIFIERS), Arrays.asList(
                        new JsonObject()
                                .with(StringUtils.capitalize(BidModifierAddMap.DEMOGRAPHICS_ADJUSTMENT),
                                        Arrays.asList(new JsonObject()
                                                .with(StringUtils.capitalize(DemographicsAdjustmentMap.GENDER),
                                                        genderValue)
                                                .with(StringUtils.capitalize(DemographicsAdjustmentMap.BID_MODIFIER),
                                                        BidModifierAddMap.DEFAULT_BID_MODIFIER)))
                                .with(StringUtils.capitalize(BidModifierAddMap.ADGROUP_ID), groupID)
                ));

        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorJsonOn(
                Action.ADD,
                addRequest,
                new Api5Error(8000,
                        Api5ErrorDetails.INCORRECT_ENUMERATE_VALUE,
                        StringUtils.capitalize(DemographicsAdjustmentMap.GENDER)),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.INCORRECT_ENUMERATE_VALUE,
                        "BidModifiers.DemographicsAdjustments." + StringUtils
                                .capitalize(DemographicsAdjustmentMap.GENDER),
                        "GENDER_MALE, GENDER_FEMALE"));
    }

    @Test
    public void unknownFieldInBidModifierTest() {
        String fieldName = "UnknownField";

        JsonObject addRequest = new JsonObject()
                .with(StringUtils.capitalize(AddRequestMap.BID_MODIFIERS), Arrays.asList(
                        new JsonObject()
                                .with(StringUtils.capitalize(BidModifierAddMap.MOBILE_ADJUSTMENT), new JsonObject()
                                        .with(StringUtils.capitalize(MobileAdjustmentMap.BID_MODIFIER),
                                                BidModifierAddMap.DEFAULT_BID_MODIFIER))
                                .with(StringUtils.capitalize(BidModifierAddMap.ADGROUP_ID), groupID)
                                .with(StringUtils.capitalize(fieldName), 123456)
                ));

        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorJsonOn(
                Action.ADD,
                addRequest,
                new Api5Error(8000,
                        Api5ErrorDetails.ELEMENT_OF_ARRAY_CONTAIN_UNKNOWN_FIELD,
                        StringUtils.capitalize(AddRequestMap.BID_MODIFIERS),
                        StringUtils.capitalize(fieldName)),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.ELEMENT_OF_ARRAY_CONTAIN_UNKNOWN_FIELD,
                        StringUtils.capitalize(AddRequestMap.BID_MODIFIERS),
                        StringUtils.capitalize(fieldName)));
    }

    @Test
    public void unknownFieldInDemographicsAdjustmentTest() {
        String fieldName = "UnknownField";

        JsonObject addRequest = new JsonObject()
                .with(StringUtils.capitalize(AddRequestMap.BID_MODIFIERS), Arrays.asList(
                        new JsonObject()
                                .with(StringUtils.capitalize(BidModifierAddMap.DEMOGRAPHICS_ADJUSTMENT),
                                        Arrays.asList(new JsonObject()
                                                .with(StringUtils.capitalize(fieldName), 123456)
                                                .with(StringUtils.capitalize(DemographicsAdjustmentMap.AGE),
                                                        AgeRangeEnum.AGE_25_34)
                                                .with(StringUtils.capitalize(DemographicsAdjustmentMap.BID_MODIFIER),
                                                        BidModifierAddMap.DEFAULT_BID_MODIFIER)))
                                .with(StringUtils.capitalize(BidModifierAddMap.ADGROUP_ID), groupID)
                ));

        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorJsonOn(
                Action.ADD,
                addRequest,
                new Api5Error(8000,
                        Api5ErrorDetails.ELEMENT_OF_ARRAY_CONTAIN_UNKNOWN_FIELD,
                        StringUtils.capitalize(BidModifierAddMap.DEMOGRAPHICS_ADJUSTMENT),
                        StringUtils.capitalize(fieldName)),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.ELEMENT_OF_ARRAY_CONTAIN_UNKNOWN_FIELD,
                        "BidModifiers." + StringUtils.capitalize(BidModifierAddMap.DEMOGRAPHICS_ADJUSTMENT),
                        StringUtils.capitalize(fieldName)));
    }
}
