package ru.yandex.autotests.direct.api.bidmodifiers.add.demographics;

import java.util.List;

import com.yandex.direct.api.v5.bidmodifiers.BidModifierFieldEnum;
import com.yandex.direct.api.v5.bidmodifiers.BidModifierGetItem;
import com.yandex.direct.api.v5.bidmodifiers.DemographicsAdjustmentFieldEnum;
import com.yandex.direct.api.v5.general.AgeRangeEnum;
import com.yandex.direct.api.v5.general.GenderEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifiersSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.DemographicsAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.MultiIdsExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.startsWith;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 17.07.2015.
 */
@Aqua.Test
@Tag(TagDictionary.TRUNK)
@Features(BidModifiersFeatures.ADD)
@Description("Позитивные сценарии добавления демографических корректировок ставок")
public class AddBidModifiersDemographicsTest {
    private static final String client = BidModifiersLogins.SINGLE_CLIENT_ADD;

    private static Long campaignId;
    private static Long adGroupId;

    private final String ID_PREFIX = "11";

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Before
    @Description("Подготовим данные для теста")
    public void createCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
    }

    @Test
    @Description("Указываем только обязательные поля демографической корректировки")
    public void onlyRequiredFieldsTest() {
        List<Long> bmIds = api.userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                        .withAge(AgeRangeEnum.AGE_25_34)
                        .withBidModifier(BidModifierAddMap.DEFAULT_BID_MODIFIER))
                .withCampaignId(campaignId));

        assertThat("добавлена демографическая корректировка ставок", bmIds, hasSize(1));
    }

    @Test
    @Description("Добавим демографическую корректировку для группы")
    public void bidModifierToGroupTest() {
        Long groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        List<Long> bmIds = api.userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                        .defaultDemographicsAdjustment())
                .withAdGroupId(groupID));

        assertThat("добавлена демографическая корректировка ставок", bmIds, hasSize(1));
    }

    @Test
    @Description("Добавим демографическую корректировку для кампании")
    public void bidModifierToCampaignTest() {
        List<Long> bmIds = api.userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                        .defaultDemographicsAdjustment())
                .withCampaignId(campaignId));

        assertThat("добавлена демографическая корректировка ставок", bmIds, hasSize(1));
    }

    @Test
    @Description("Добавим несколько демографических корректировок для кампании")
    public void multipleDemographicsAdjustmentToCampaignTest() {
        List<Long> bmIds = api.userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                .withGender(GenderEnum.GENDER_MALE)
                                .withAge(AgeRangeEnum.AGE_18_24)
                                .withDefaultBidModifier(),
                        new DemographicsAdjustmentMap()
                                .withGender(GenderEnum.GENDER_FEMALE)
                                .withAge(AgeRangeEnum.AGE_18_24)
                                .withDefaultBidModifier())
                .withCampaignId(campaignId));

        assertThat("добавлены демографические корректировки ставок", bmIds, hasSize(2));
    }

    @Test
    @Description("Добавим несколько демографических корректировок для группы")
    public void multipleDemographicsAdjustmentToGroupTest() {
        Long groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        List<Long> bmIds = api.userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                .withGender(GenderEnum.GENDER_MALE)
                                .withAge(AgeRangeEnum.AGE_18_24)
                                .withDefaultBidModifier(),
                        new DemographicsAdjustmentMap()
                                .withGender(GenderEnum.GENDER_FEMALE)
                                .withAge(AgeRangeEnum.AGE_18_24)
                                .withDefaultBidModifier())
                .withAdGroupId(groupID));

        assertThat("добавлены демографические корректировки ставок", bmIds, hasSize(2));
    }

    @Test
    @Description("Добавим одинаковые демографические корректировоки для кампании и группы")
    public void sameDemographicsAdjustmentToGroupAndCampaignTest() {
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        Long groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        List<Long> bmIds = api.userSteps.bidModifiersSteps().bidModifiersAdd(
                new BidModifierAddMap()
                        .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                .defaultDemographicsAdjustment())
                        .withAdGroupId(groupID),
                new BidModifierAddMap()
                        .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                .defaultDemographicsAdjustment())
                        .withCampaignId(campaignID.longValue())
        );

        assertThat("добавлены идентичные демографические корректировки в группу и кампанию", bmIds, hasSize(2));
    }

    @Test
    @Description("Идентификатор демографической корректировки ставки должен начинаться с префикса - " + ID_PREFIX)
    public void complexIdentificatorDemographicsTest() {
        Long groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        List<Long> bmIds = api.userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                        .defaultDemographicsAdjustment())
                .withAdGroupId(groupID));
        assumeThat("добавлена демографическая корректировка ставки", bmIds, hasSize(1));

        assertThat("идентификатор демографической корректировки ставки начинается с корректного префикса",
                String.valueOf(bmIds.get(0)), startsWith(ID_PREFIX));
    }

    @Test
    @Description("Добавим корректировоку AGE_45_54")
    public void demographicsAdjustmentAge45_54Test() {
        List<Long> bmIds = api.userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                        .withAge(AgeRangeEnum.AGE_45_54)
                        .withBidModifier(BidModifierAddMap.DEFAULT_BID_MODIFIER))
                .withCampaignId(campaignId));

        List<BidModifierGetItem> bms = api.userSteps.bidModifiersSteps().bidModifiersGet(
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap()
                                .withIds(bmIds.toArray(new Long[0]))
                                .withAllLevels())
                        .withFieldNames(BidModifierFieldEnum.ID)
                        .withDemographicsFieldNames(DemographicsAdjustmentFieldEnum.AGE)
        );

        assertThat("поле Age сохранилось правильно",
                bms.get(0).getDemographicsAdjustment().getAge().getValue(), equalTo(AgeRangeEnum.AGE_45_54)
        );
    }

    @Test
    @Description("Добавим корректировоку AGE_45")
    public void demographicsAdjustmentAge45Test() {
        List<Long> bmIds = api.userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                        .withAge(AgeRangeEnum.AGE_45)
                        .withBidModifier(BidModifierAddMap.DEFAULT_BID_MODIFIER))
                .withCampaignId(campaignId));

        List<BidModifierGetItem> bms = api.userSteps.bidModifiersSteps().bidModifiersGet(
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap()
                                .withIds(bmIds.toArray(new Long[0]))
                                .withAllLevels())
                        .withFieldNames(BidModifierFieldEnum.ID)
                        .withDemographicsFieldNames(DemographicsAdjustmentFieldEnum.AGE)
        );

        assertThat("поле Age сохранилось правильно",
                bms.get(0).getDemographicsAdjustment().getAge().getValue(), equalTo(AgeRangeEnum.AGE_45)
        );
    }

    @Test
    @Description("Добавим пересекающиеся корректировоки в кампанию и группу обьявлений")
    public void sameDemographicsAdjustmentForCampaignAndAdGroup() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(new BidModifierAddMap().withDemographicsAdjustment(
                        new DemographicsAdjustmentMap()
                                .withAge(AgeRangeEnum.AGE_45)
                                .withBidModifier(120))
                        .withCampaignId(campaignId.longValue())),
                MultiIdsExpectedResult.success());
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(new BidModifierAddMap().withDemographicsAdjustment(
                        new DemographicsAdjustmentMap()
                                .withAge(AgeRangeEnum.AGE_55)
                                .withBidModifier(120))
                        .withAdGroupId(adGroupId.longValue())),
                MultiIdsExpectedResult.success());
    }
}
