package ru.yandex.autotests.direct.api.bidmodifiers.add.desktop;

import java.util.List;

import com.google.common.primitives.Shorts;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.HierarchicalMultipliersRecord;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.DesktopAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.general.MultiIdsExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasProperty;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.startsWith;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(BidModifiersFeatures.ADD)
@Tag(TagDictionary.TRUNK)
@Description("Позитивные сценарии добавления дектопных корректировок")
public class AddBidModifiersDesktopTest {
    private static final String LOGIN = BidModifiersLogins.SINGLE_CLIENT_ADD;
    private static final String DESKTOP_ID_PREFIX = "19";

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private static Long campaignId;
    private Long adGroupId;

    @BeforeClass
    public static void beforeClass() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        campaignId = api.userSteps.campaignSteps().addDefaultCpmBannerCampaign();
    }

    @Before
    public void before() {
        adGroupId = api.userSteps.adGroupsSteps().addDefaultCpmBannerKeywordsAdGroup(campaignId);
    }

    @Test
    public void addAdjustmentAndCheckTypeInDatabase() {
        HierarchicalMultipliersRecord dbRecord =
                createDesktopAdjustmentAndGetDbRecord(BidModifierAddMap.DEFAULT_BID_MODIFIER);
        assertThat("Запись в БД создана правильно",
                dbRecord,
                allOf(
                        hasProperty("cid", equalTo(campaignId)),
                        hasProperty("pid", equalTo(adGroupId)),
                        hasProperty("type", equalTo(HierarchicalMultipliersType.desktop_multiplier)),
                        hasProperty("multiplierPct",
                                equalTo(Shorts.checkedCast(BidModifierAddMap.DEFAULT_BID_MODIFIER))),
                        hasProperty("isEnabled", equalTo(1))
                )
        );
    }

    @Test
    public void addMinimalAdjustmentAndCheckValueInDatabase() {
        HierarchicalMultipliersRecord dbRecord =
                createDesktopAdjustmentAndGetDbRecord(DesktopAdjustmentMap.DESKTOP_ADJUSTMENT_ABSOLUTE_MIN);

        assertThat("Запись в БД создана с правильным коэффициентом",
                dbRecord.getMultiplierPct(),
                equalTo(Shorts.checkedCast(DesktopAdjustmentMap.DESKTOP_ADJUSTMENT_ABSOLUTE_MIN)));
    }

    @Test
    public void addMaximumAdjustmentAndCheckValueInDatabase() {
        HierarchicalMultipliersRecord dbRecord =
                createDesktopAdjustmentAndGetDbRecord(DesktopAdjustmentMap.DESKTOP_ADJUSTMENT_MAX);

        assertThat("Запись в БД создана с правильным коэффициентом",
                dbRecord.getMultiplierPct(), equalTo(Shorts.checkedCast(DesktopAdjustmentMap.DESKTOP_ADJUSTMENT_MAX)));
    }

    @Test
    public void checkIdentifierPrefixIsCorrect() {
        List<Long> apiIds = api.userSteps.bidModifiersSteps().bidModifiersAdd(
                new BidModifierAddMap()
                        .withAdGroupId(adGroupId)
                        .withDesktopAdjustment(new DesktopAdjustmentMap().defaultDesktopAdjustment())
        );
        assumeThat("добавлена корректировка ставки", apiIds, hasSize(1));
        assertThat("идентификатор корректировки ставки начинается с корректного префикса",
                String.valueOf(apiIds.get(0)), startsWith(DESKTOP_ID_PREFIX));
    }

    private HierarchicalMultipliersRecord createDesktopAdjustmentAndGetDbRecord(int bidModifierMultiplierPct) {
        api.userSteps.bidModifiersSteps().shouldGetResultOnAdd(
                new AddRequestMap().withBidModifiers(
                        new BidModifierAddMap()
                                .withAdGroupId(adGroupId)
                                .withDesktopAdjustment(
                                        new DesktopAdjustmentMap().withBidModifier(bidModifierMultiplierPct))
                ),
                MultiIdsExpectedResult.success()
        );
        return api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN)
                .multipliersSteps().getHierarchicalMultipliersByPid(adGroupId)
                .get(0);
    }
}
