package ru.yandex.autotests.direct.api.bidmodifiers.add.mobile;

import java.math.BigInteger;
import java.util.List;

import com.google.common.primitives.Shorts;
import com.yandex.direct.api.v5.bidmodifiers.OperatingSystemTypeEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.MobileMultiplierValuesOsType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.HierarchicalMultipliersRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.MobileMultiplierValuesRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.MobileAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.general.MultiIdsExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasProperty;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(BidModifiersFeatures.ADD)
@Description("Проверка состояния БД при добавлении мобильных корректировок c таргетингом на ОС")
public class AddBidModifiersMobileWithOsTargetingAndCheckDbTest {
    private static final String LOGIN = BidModifiersLogins.SINGLE_CLIENT_ADD;

    private static final String FIELD_CID = "cid";
    private static final String FIELD_PID = "pid";
    private static final String FIELD_TYPE = "type";
    private static final String FIELD_MULTIPLIER_PCT = "multiplierPct";
    private static final String FIELD_IS_ENABLED = "isEnabled";
    private static final String FIELD_OS_TYPE = "osType";
    private static final String FIELD_HIERARCHICAL_MULTIPLIER_ID = "hierarchicalMultiplierId";
    private static final String FIELD_LAST_CHANGE = "lastChange";

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private static Long campaignId;
    private static DirectJooqDbSteps jooqDbSteps;

    private Long adGroupId;

    @BeforeClass
    public static void beforeClass() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN);
        campaignId = api.userSteps.campaignSteps().addDefaultCpmBannerCampaign();
    }

    @Before
    public void before() {
        adGroupId = api.userSteps.adGroupsSteps().addDefaultCpmBannerKeywordsAdGroup(campaignId);
    }

    @Test
    public void addAdjustmentWithoutOsTypeAndCheckRecordInHierarchicalMultipliersTable() {
        HierarchicalMultipliersRecord dbRecord = createMobileAdjustment(new MobileAdjustmentMap()
                .withOperatingSystemType(null)
                .withBidModifier(BidModifierAddMap.DEFAULT_BID_MODIFIER));

        assertThat("Запись в hierarchical_multipliers создана правильно",
                dbRecord,
                allOf(
                        hasProperty(FIELD_CID, equalTo(campaignId)),
                        hasProperty(FIELD_PID, equalTo(adGroupId)),
                        hasProperty(FIELD_TYPE, equalTo(HierarchicalMultipliersType.mobile_multiplier)),
                        hasProperty(FIELD_MULTIPLIER_PCT,
                                equalTo(Shorts.checkedCast(BidModifierAddMap.DEFAULT_BID_MODIFIER))),
                        hasProperty(FIELD_IS_ENABLED, equalTo(1)),
                        hasProperty(FIELD_LAST_CHANGE, notNullValue())
                )
        );
    }

    @Test
    public void addIosAdjustmentAndCheckRecordInHierarchicalMultipliersTable() {
        HierarchicalMultipliersRecord dbRecord = createMobileAdjustment(new MobileAdjustmentMap()
                .withOperatingSystemType(OperatingSystemTypeEnum.IOS)
                .withBidModifier(BidModifierAddMap.DEFAULT_BID_MODIFIER));

        assertThat("Запись в hierarchical_multipliers создана правильно",
                dbRecord,
                allOf(
                        hasProperty(FIELD_CID, equalTo(campaignId)),
                        hasProperty(FIELD_PID, equalTo(adGroupId)),
                        hasProperty(FIELD_TYPE, equalTo(HierarchicalMultipliersType.mobile_multiplier)),
                        hasProperty(FIELD_IS_ENABLED, equalTo(1)),
                        hasProperty(FIELD_LAST_CHANGE, notNullValue()),
                        // для корректировки указана ОС, поэтому значение в отдельной таблице
                        hasProperty(FIELD_MULTIPLIER_PCT, nullValue())
                )
        );
    }

    @Test
    public void addAndroidAdjustmentAndCheckRecordInHierarchicalMultipliersTable() {
        HierarchicalMultipliersRecord dbRecord = createMobileAdjustment(new MobileAdjustmentMap()
                .withOperatingSystemType(OperatingSystemTypeEnum.ANDROID)
                .withBidModifier(BidModifierAddMap.DEFAULT_BID_MODIFIER));

        assertThat("Запись в hierarchical_multipliers создана правильно",
                dbRecord,
                allOf(
                        hasProperty(FIELD_CID, equalTo(campaignId)),
                        hasProperty(FIELD_PID, equalTo(adGroupId)),
                        hasProperty(FIELD_TYPE, equalTo(HierarchicalMultipliersType.mobile_multiplier)),
                        hasProperty(FIELD_IS_ENABLED, equalTo(1)),
                        hasProperty(FIELD_LAST_CHANGE, notNullValue()),
                        // для корректировки указана ОС, поэтому значение в отдельной таблице
                        hasProperty(FIELD_MULTIPLIER_PCT, nullValue())
                )
        );
    }

    @Test
    public void addAdjustmentWithoutOsTypeAndCheckRecordInMobileMultiplierValuesTable() {
        createMobileAdjustment(new MobileAdjustmentMap()
                .withOperatingSystemType(null)
                .withBidModifier(BidModifierAddMap.DEFAULT_BID_MODIFIER));

        List<MobileMultiplierValuesRecord> mobileMultiplierValues = jooqDbSteps.multipliersSteps()
                .getMobileMultiplierValuesByAdGroupId(adGroupId);
        assertThat(
                "Для мобильной корректировки без указания ОС не должны создаваться записи в mobile_multiplier_values",
                mobileMultiplierValues, empty());
    }

    @Test
    public void addIosAdjustmentAndCheckRecordInMobileMultiplierValuesTable() {
        BigInteger hierarchicalMultiplierId = createMobileAdjustment(new MobileAdjustmentMap()
                .withOperatingSystemType(OperatingSystemTypeEnum.IOS)
                .withBidModifier(BidModifierAddMap.DEFAULT_BID_MODIFIER))
                .getHierarchicalMultiplierId();

        List<MobileMultiplierValuesRecord> mobileMultiplierValues = jooqDbSteps.multipliersSteps()
                .getMobileMultiplierValuesByAdGroupId(adGroupId);

        assertThat("Запись в mobile_multiplier_values создана правильно",
                mobileMultiplierValues,
                contains(
                        allOf(
                                hasProperty(FIELD_HIERARCHICAL_MULTIPLIER_ID, equalTo(hierarchicalMultiplierId)),
                                hasProperty(FIELD_OS_TYPE, equalTo(MobileMultiplierValuesOsType.ios)),
                                hasProperty(FIELD_MULTIPLIER_PCT,
                                        equalTo(Shorts.checkedCast(BidModifierAddMap.DEFAULT_BID_MODIFIER))),
                                hasProperty(FIELD_LAST_CHANGE, notNullValue())
                        )
                )
        );
    }

    @Test
    public void addAndroidAdjustmentAndCheckRecordInMobileMultiplierValuesTable() {
        BigInteger hierarchicalMultiplierId = createMobileAdjustment(new MobileAdjustmentMap()
                .withOperatingSystemType(OperatingSystemTypeEnum.ANDROID)
                .withBidModifier(BidModifierAddMap.DEFAULT_BID_MODIFIER))
                .getHierarchicalMultiplierId();

        List<MobileMultiplierValuesRecord> mobileMultiplierValues = jooqDbSteps.multipliersSteps()
                .getMobileMultiplierValuesByAdGroupId(adGroupId);

        assertThat("Запись в mobile_multiplier_values создана правильно",
                mobileMultiplierValues,
                contains(
                        allOf(
                                hasProperty(FIELD_HIERARCHICAL_MULTIPLIER_ID, equalTo(hierarchicalMultiplierId)),
                                hasProperty(FIELD_OS_TYPE, equalTo(MobileMultiplierValuesOsType.android)),
                                hasProperty(FIELD_MULTIPLIER_PCT,
                                        equalTo(Shorts.checkedCast(BidModifierAddMap.DEFAULT_BID_MODIFIER))),
                                hasProperty(FIELD_LAST_CHANGE, notNullValue())
                        )
                )
        );
    }

    @Test
    public void addMinimalAndroidAdjustmentAndCheckValueInDatabase() {
        createMobileAdjustment(new MobileAdjustmentMap()
                .withOperatingSystemType(OperatingSystemTypeEnum.ANDROID)
                .withBidModifier(MobileAdjustmentMap.MOBILE_ADJUSTMENT_ABSOLUTE_MIN));

        List<MobileMultiplierValuesRecord> mobileMultiplierValues = jooqDbSteps.multipliersSteps()
                .getMobileMultiplierValuesByAdGroupId(adGroupId);

        assertThat("Запись в mobile_multiplier_values с правильным коэффициентом",
                mobileMultiplierValues,
                contains(hasProperty(FIELD_MULTIPLIER_PCT,
                        equalTo(Shorts.checkedCast(MobileAdjustmentMap.MOBILE_ADJUSTMENT_ABSOLUTE_MIN))))
        );
    }

    @Test
    public void addMaximumAndroidAdjustmentAndCheckValueInDatabase() {
        createMobileAdjustment(new MobileAdjustmentMap()
                .withOperatingSystemType(OperatingSystemTypeEnum.ANDROID)
                .withBidModifier(MobileAdjustmentMap.MOBILE_ADJUSTMENT_MAX));

        List<MobileMultiplierValuesRecord> mobileMultiplierValues = jooqDbSteps.multipliersSteps()
                .getMobileMultiplierValuesByAdGroupId(adGroupId);

        assertThat("Запись в mobile_multiplier_values с правильным коэффициентом",
                mobileMultiplierValues,
                contains(hasProperty(FIELD_MULTIPLIER_PCT,
                        equalTo(Shorts.checkedCast(MobileAdjustmentMap.MOBILE_ADJUSTMENT_MAX))))
        );
    }

    @Test
    public void addMinimalAdjustmentWithoutOsAndCheckValueInDatabase() {
        HierarchicalMultipliersRecord dbRecord = createMobileAdjustment(new MobileAdjustmentMap()
                .withOperatingSystemType(null)
                .withBidModifier(MobileAdjustmentMap.MOBILE_ADJUSTMENT_ABSOLUTE_MIN));

        assertThat("Запись в hierarchical_multipliers с правильным коэффициентом",
                dbRecord,
                hasProperty(FIELD_MULTIPLIER_PCT,
                        equalTo(Shorts.checkedCast(MobileAdjustmentMap.MOBILE_ADJUSTMENT_ABSOLUTE_MIN)))
        );
    }

    @Test
    public void addMaximumAdjustmentWithoutOsAndCheckValueInDatabase() {
        HierarchicalMultipliersRecord dbRecord = createMobileAdjustment(new MobileAdjustmentMap()
                .withOperatingSystemType(null)
                .withBidModifier(MobileAdjustmentMap.MOBILE_ADJUSTMENT_MAX));

        assertThat("Запись в hierarchical_multipliers с правильным коэффициентом",
                dbRecord,
                hasProperty(FIELD_MULTIPLIER_PCT,
                        equalTo(Shorts.checkedCast(MobileAdjustmentMap.MOBILE_ADJUSTMENT_MAX)))
        );
    }

    private HierarchicalMultipliersRecord createMobileAdjustment(MobileAdjustmentMap mobileAdjustment) {
        api.userSteps.bidModifiersSteps().shouldGetResultOnAdd(
                new AddRequestMap().withBidModifiers(
                        new BidModifierAddMap()
                                .withAdGroupId(adGroupId)
                                .withMobileAdjustment(mobileAdjustment)
                ),
                MultiIdsExpectedResult.success()
        );
        List<HierarchicalMultipliersRecord> hierarchicalMultipliersByPid =
                jooqDbSteps.multipliersSteps().getHierarchicalMultipliersByPid(adGroupId);
        // тесты расчитаны на то, что в одну группу не добавляется несколько корректировок
        assumeThat("Создана одна соотвествующая группе запись", hierarchicalMultipliersByPid, hasSize(1));
        return hierarchicalMultipliersByPid.get(0);
    }
}
