package ru.yandex.autotests.direct.api.bidmodifiers.add.regional;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.utils.model.RegionIDValues;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.RegionalAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.general.MultiIdsExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;


/**
 * https://st.yandex-team.ru/DIRECT-64099
 */
@Aqua.Test
@Features(BidModifiersFeatures.ADD)
@Issue("https://st.yandex-team.ru/DIRECT-63882")
@Description("Негативные сценарии добавления региональных корректировок ставок")
public class AddBidModifiersRegionalNegativeTest {
    private static final String LOGIN = BidModifiersLogins.SINGLE_CLIENT_ADD;
    private static final Long NONEXISTENT_REGION_ID = 123456L;
    private static final Long ANOTHER_NONEXISTENT_REGION_ID = 654321L;
    private static final Long INVALID_REGION_ID = -213L;

    private static Long campaignId;
    private static Long anotherCampaingId;
    private static Long groupId;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void createCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        anotherCampaingId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        groupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
    }


    @Test
    @Description("Добавим две одинаковые корректировки двумя запросами")
    public void sameRegionalAdjustmentsInSomeRequestTest() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(new BidModifierAddMap()
                        .withRegionalAdjustment(
                                new RegionalAdjustmentMap().defaultRegionalAdjustment())
                        .withCampaignId(campaignId)),
                MultiIdsExpectedResult.success());
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(new BidModifierAddMap()
                        .withRegionalAdjustment(
                                new RegionalAdjustmentMap().defaultRegionalAdjustment())
                        .withCampaignId(campaignId)),
                MultiIdsExpectedResult.errors(new Notification(6000, Api5ErrorDetails.BID_MODIFIER_REGIONAL_ADJUSTMENT_INTERSECTION)));
    }

    @Test
    @Description("Добавим две одинаковые корректировки одним запросом")
    public void sameRegionalAdjustmentsInOneRequestTest() {
        api.userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withRegionalAdjustment(
                        new RegionalAdjustmentMap()
                                .defaultRegionalAdjustment()
                                .withRegionId(RegionIDValues.UKRAINE.getId()))
                .withCampaignId(campaignId));
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(new BidModifierAddMap()
                        .withRegionalAdjustment(
                                new RegionalAdjustmentMap()
                                        .defaultRegionalAdjustment()
                                        .withRegionId(RegionIDValues.UKRAINE.getId()))
                        .withCampaignId(campaignId)),
                MultiIdsExpectedResult.errors(new Notification(6000, Api5ErrorDetails.BID_MODIFIER_REGIONAL_ADJUSTMENT_INTERSECTION)));
    }

    @Test
    @Description("Добавляем корректировку ставок на группу")
    public void regionalAdjustmentToGroupTest() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(new BidModifierAddMap()
                        .withRegionalAdjustment(
                                new RegionalAdjustmentMap().defaultRegionalAdjustment())
                        .withAdGroupId(groupId)),
                singletonList(MultiIdsExpectedResult.errors(new Notification(3500, Api5ErrorDetails.REGIONAL_ADJUSTMENT_NOT_SUPPORTED_IN_ADGROUP))),
                singletonList(MultiIdsExpectedResult.errors(new Notification(3500, Api5ErrorDetailsJava.REGIONAL_ADJUSTMENT_NOT_SUPPORTED_IN_ADGROUP))));
    }

    @Test
    public void regionalAdjustmentWithNonexistentRegion() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(new BidModifierAddMap()
                        .withRegionalAdjustment(
                                new RegionalAdjustmentMap().defaultRegionalAdjustment()
                                        .withRegionId(NONEXISTENT_REGION_ID))
                        .withCampaignId(campaignId)),
                MultiIdsExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetailsJava.NONEXISTENT_REGION_IDS, NONEXISTENT_REGION_ID.toString())));
    }

    @Test
    public void regionalAdjustmentWithInvalidRegion() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(new BidModifierAddMap()
                        .withRegionalAdjustment(
                                new RegionalAdjustmentMap().defaultRegionalAdjustment()
                                        .withRegionId(INVALID_REGION_ID))
                        .withCampaignId(campaignId)),
                MultiIdsExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetailsJava.NONEXISTENT_REGION_IDS, INVALID_REGION_ID.toString())));
    }

    @Test
    public void regionalAdjustmentsWithValidAndNonexistentRegion() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(new BidModifierAddMap()
                        .withRegionalAdjustment(
                                new RegionalAdjustmentMap().defaultRegionalAdjustment()
                                        .withRegionId(NONEXISTENT_REGION_ID),
                                new RegionalAdjustmentMap().defaultRegionalAdjustment())
                        .withCampaignId(campaignId)),
                MultiIdsExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetailsJava.NONEXISTENT_REGION_IDS, NONEXISTENT_REGION_ID.toString())));
    }

    @Test
    public void regionalAdjustmentsWithValidAndNonexistentRegionInDifferentItems() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(
                        new BidModifierAddMap()
                                .withRegionalAdjustment(new RegionalAdjustmentMap()
                                        .defaultRegionalAdjustment()
                                        .withRegionId(NONEXISTENT_REGION_ID))
                        .withCampaignId(campaignId),
                        new BidModifierAddMap()
                                .withRegionalAdjustment(new RegionalAdjustmentMap().defaultRegionalAdjustment())
                                .withCampaignId(anotherCampaingId)),
                MultiIdsExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetailsJava.NONEXISTENT_REGION_IDS, NONEXISTENT_REGION_ID.toString())),
                MultiIdsExpectedResult.success());
    }

    @Test
    public void regionalAdjustmentsWithTwoNonexistentRegions() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withBidModifiers(new BidModifierAddMap()
                        .withRegionalAdjustment(
                                new RegionalAdjustmentMap().defaultRegionalAdjustment()
                                        .withRegionId(NONEXISTENT_REGION_ID),
                                new RegionalAdjustmentMap().defaultRegionalAdjustment()
                                        .withRegionId(ANOTHER_NONEXISTENT_REGION_ID))
                        .withCampaignId(campaignId)),
                MultiIdsExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetailsJava.NONEXISTENT_REGION_IDS,
                        String.join(",",
                                asList(NONEXISTENT_REGION_ID.toString(), ANOTHER_NONEXISTENT_REGION_ID.toString())))));
    }
}
