package ru.yandex.autotests.direct.api.bidmodifiers.add.regional;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.GeoMultiplierValuesRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.HierarchicalMultipliersRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.model.RegionIDValues;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.RegionalAdjustmentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.directapi.model.api5.bidmodifiers.RegionalAdjustmentMap.REGIONAL_ADJUSTMENT_MAX;
import static ru.yandex.autotests.directapi.model.api5.bidmodifiers.RegionalAdjustmentMap.REGIONAL_ADJUSTMENT_MIN;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(BidModifiersFeatures.ADD)
@RunWith(Parameterized.class)
@Issue("https://st.yandex-team.ru/DIRECT-67328")
@Description("Проверка неявного добавления скрытой корректировки для Крыма")
public class AddHiddenBidModifiersRegionalForCrimeaTest {

    private static final Integer modifierNotHidden = 0;
    private static final Integer modifierIsHidden = 1;

    private static final Long russiaGeoId = Long.valueOf(RegionIDValues.RUSSIA.getId());
    private static final Long ukraineGeoId = Long.valueOf(RegionIDValues.UKRAINE.getId());
    private static final Long moscowGeoId = Long.valueOf(RegionIDValues.MOSCOW.getId());
    private static final Long kievGeoId = Long.valueOf(RegionIDValues.KIEV.getId());
    private static final Long crimeaGeoId = Long.valueOf(RegionIDValues.CRIMEA.getId());

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    @Parameterized.Parameter
    public String description;

    @Parameterized.Parameter(1)
    public String client;

    @Parameterized.Parameter(2)
    public Long homeCountryRegionId;

    @Parameterized.Parameter(3)
    public Long homeCapitalRegionId;

    @Parameterized.Parameter(4)
    public Long abroadRegionId;

    private Long campaignId;
    private DirectJooqDbSteps jooqSteps;

    @Parameterized.Parameters(name = "{0}")
    public static Collection testData() {
        Object[][] data = new Object[][]{
                {"Клиент из России", BidModifiersLogins.SINGLE_CLIENT_ADD, russiaGeoId, moscowGeoId, ukraineGeoId},
                {"Клиент из Украины", Logins.LOGIN_FOR_UAH, ukraineGeoId, kievGeoId, russiaGeoId},
        };
        return Arrays.asList(data);
    }

    @Before
    public void prepare() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(client);

        jooqSteps = api.userSteps.getDirectJooqDbSteps();
        jooqSteps.useShardForLogin(client);
    }

    @Test
    @Description("Клиент ставит корректировку на свою страну - должна появиться скрытая корректировка на Крым с таким же мультипликатором")
    public void clientSetModifierForHomeParentRegion() {
        List<Long> bmIds = api.as(client).userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withRegionalAdjustment(new RegionalAdjustmentMap().withBidModifier(REGIONAL_ADJUSTMENT_MIN)
                        .withRegionId(homeCountryRegionId)).withCampaignId(campaignId));
        assumeThat("добавлена 1 гео корректировка", bmIds, hasSize(1));

        List<HierarchicalMultipliersRecord> multipliers =
                jooqSteps.multipliersSteps().getHierarchicalMultipliersByCid(campaignId);
        assumeThat("в бд 1 иерархическая корректировка", multipliers, hasSize(1));

        List<GeoMultiplierValuesRecord> geoMultipliers =
                jooqSteps.multipliersSteps().getGeoMultiplierValuesByHierarchicalMultipliersId(
                        multipliers.get(0).getHierarchicalMultiplierId().longValue());
        assumeThat("в бд 2 гео корректировки", geoMultipliers, hasSize(2));
        assumeThat("мультипликатор корректировки для Крыма ожидаемый",
                geoMultipliers.stream().filter(e -> e.getRegionId().equals(crimeaGeoId))
                        .map(e -> Integer.valueOf(e.getMultiplierPct())).collect(Collectors.toList()),
                contains(equalTo(REGIONAL_ADJUSTMENT_MIN)));
        assertThat("скрытая корректировка для Крыма",
                geoMultipliers.stream().filter(e -> e.getRegionId().equals(crimeaGeoId))
                        .map(GeoMultiplierValuesRecord::getIsHidden).collect(Collectors.toList()),
                contains(equalTo(modifierIsHidden)));
    }

    @Test
    @Description("Клиент ставит корректировку на дочерний регион своей страны (столицу) - скрытая корректировка на Крым не должна появиться")
    public void clientSetModifierForChildRegionOfHomeRegion() {
        List<Long> bmIds = api.as(client).userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withRegionalAdjustment(
                        new RegionalAdjustmentMap().withBidModifier(REGIONAL_ADJUSTMENT_MIN)
                                .withRegionId(homeCapitalRegionId))
                .withCampaignId(campaignId));
        assumeThat("добавлена 1 гео корректировка", bmIds, hasSize(1));

        List<HierarchicalMultipliersRecord> multipliers =
                jooqSteps.multipliersSteps().getHierarchicalMultipliersByCid(campaignId);
        assumeThat("в бд 1 иерархическая корректировка", multipliers, hasSize(1));

        List<GeoMultiplierValuesRecord> geoMultipliers =
                jooqSteps.multipliersSteps().getGeoMultiplierValuesByHierarchicalMultipliersId(
                        multipliers.get(0).getHierarchicalMultiplierId().longValue());
        assertThat("в бд 1 гео корректировка", geoMultipliers, hasSize(1));
    }

    @Test
    @Description("Клиент ставит корректировку на соседнюю страну - скрытая корректировка на Крым не должна появиться")
    public void clientSetModifierForAbroadCountry() {
        List<Long> bmIds = api.as(client).userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withRegionalAdjustment(new RegionalAdjustmentMap().withBidModifier(REGIONAL_ADJUSTMENT_MIN)
                        .withRegionId(abroadRegionId)).withCampaignId(campaignId));
        assumeThat("добавлена 1 гео корректировка", bmIds, hasSize(1));

        List<HierarchicalMultipliersRecord> multipliers =
                jooqSteps.multipliersSteps().getHierarchicalMultipliersByCid(campaignId);
        assumeThat("в бд 1 иерархическая корректировка", multipliers, hasSize(1));

        List<GeoMultiplierValuesRecord> geoMultipliers =
                jooqSteps.multipliersSteps().getGeoMultiplierValuesByHierarchicalMultipliersId(
                        multipliers.get(0).getHierarchicalMultiplierId().longValue());
        assertThat("в бд 1 гео корректировка", geoMultipliers, hasSize(1));
    }

    @Test
    @Description("Клиент ставит корректировку на свою страну, затем добавляет корректировку на Крым - скрытая корректировка на Крым должна быть заменена на выставленную")
    public void clientSetModifierForCrimea() {
        List<Long> bmIds = api.as(client).userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withRegionalAdjustment(new RegionalAdjustmentMap().withBidModifier(REGIONAL_ADJUSTMENT_MIN)
                        .withRegionId(homeCountryRegionId)).withCampaignId(campaignId));
        assumeThat("добавлена 1 гео корректировка", bmIds, hasSize(1));


        List<HierarchicalMultipliersRecord> multipliers =
                jooqSteps.multipliersSteps().getHierarchicalMultipliersByCid(campaignId);
        assumeThat("в бд 1 иерархическая корректировка", multipliers, hasSize(1));

        List<GeoMultiplierValuesRecord> geoMultipliers = jooqSteps.multipliersSteps()
                .getGeoMultiplierValuesByHierarchicalMultipliersId(
                        multipliers.get(0).getHierarchicalMultiplierId().longValue());
        assumeThat("в бд 2 гео корректировки", geoMultipliers, hasSize(2));
        assumeThat("мультипликатор корректировки для Крыма ожидаемый",
                geoMultipliers.stream().filter(e -> e.getRegionId().equals(crimeaGeoId))
                        .map(e -> Integer.valueOf(e.getMultiplierPct())).collect(Collectors.toList()),
                contains(equalTo(REGIONAL_ADJUSTMENT_MIN)));
        assumeThat("скрытая корректировка для Крыма",
                geoMultipliers.stream().filter(e -> e.getRegionId().equals(crimeaGeoId))
                        .map(GeoMultiplierValuesRecord::getIsHidden).collect(Collectors.toList()),
                contains(equalTo(modifierIsHidden)));


        bmIds = api.as(client).userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withRegionalAdjustment(
                        new RegionalAdjustmentMap().withBidModifier(REGIONAL_ADJUSTMENT_MAX).withRegionId(crimeaGeoId))
                .withCampaignId(campaignId));
        assumeThat("добавлена 1 гео корректировка", bmIds, hasSize(1));

        multipliers = jooqSteps.multipliersSteps().getHierarchicalMultipliersByCid(campaignId);
        assumeThat("в бд 1 иерархическая корректировка", multipliers, hasSize(1));

        geoMultipliers = jooqSteps.multipliersSteps().getGeoMultiplierValuesByHierarchicalMultipliersId(
                multipliers.get(0).getHierarchicalMultiplierId().longValue());
        assumeThat("в бд 2 гео корректировки", geoMultipliers, hasSize(2));
        assumeThat("мультипликатор корректировки для Крыма ожидаемый",
                geoMultipliers.stream().filter(e -> e.getRegionId().equals(crimeaGeoId))
                        .map(e -> Integer.valueOf(e.getMultiplierPct())).collect(Collectors.toList()),
                contains(equalTo(REGIONAL_ADJUSTMENT_MAX)));
        assertThat("корректировка для Крыма не скрытая",
                geoMultipliers.stream().filter(e -> e.getRegionId().equals(crimeaGeoId))
                        .map(GeoMultiplierValuesRecord::getIsHidden).collect(Collectors.toList()),
                contains(equalTo(modifierNotHidden)));
    }
}
