package ru.yandex.autotests.direct.api.bidmodifiers.delete;

import java.util.List;

import com.yandex.direct.api.v5.bidmodifiers.BidModifierGetItem;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.extract;
import static ch.lambdaj.Lambda.on;
import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by chicos on 31.07.15
 */
@Aqua.Test
@Features(BidModifiersFeatures.DELETE)
@Description("Проверка негативных сценариев удаления корректировок ставок")
public class DeleteBidModifiersNegativeTest {
    private static final String client = BidModifiersLogins.SINGLE_CLIENT_DELETE;

    private static Long bmArchivedID;
    private static Long bmID;

    private static long bmID1;
    private static long bmID2;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepareBidModifiers() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        Long campaignArchived = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        bmArchivedID = api.userSteps.bidModifiersSteps().addBidModifierMobileToCampaign(campaignArchived);
        api.userSteps.campaignSteps().campaignsSuspend(campaignArchived);
        api.userSteps.campaignSteps().campaignsArchive(campaignArchived);

        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        bmID = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToCampaign(campaignID.intValue());

        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();
        long groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        bmID1 = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID);
        bmID2 = api.userSteps.bidModifiersSteps().addBidModifierMobileToGroup(groupID);
    }

    @Test
    @Description("Попробуем удалить корректировку ставки из архивной кампании")
    public void archivedCampaignTest() {
        api.userSteps.bidModifiersSteps().shouldGetEitherResultOn(
                Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(bmArchivedID)),
                singletonList(ExpectedResult.errors(new Notification(8300,
                                Api5ErrorDetails.UNABLE_UPDATE_ARCHIVED_CAMPAGN))),
                singletonList((ExpectedResult) ExpectedResult.errors(new Notification(8300,
                        Api5ErrorDetails.UNABLE_UPDATE_ARCHIVED_CAMPAGN)).withId(bmArchivedID))
        );
    }

    @Test
    @Description("Удаление двух корректировок ставок, идентификатор одной из которых - некорректен")
    public void deleteMultipleBidModifiersOneInvalidTest() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(bmID, 123456L)),
                ExpectedResult.success(bmID),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.BID_MODIFIER_NOT_FOUND)));
    }

    @Test
    @Description("Два одинаковых идентификатора корректировки ставок в запросе")
    public void sameIdsInRequest() {
        api.userSteps.bidModifiersSteps().shouldGetEitherResultOn(
                Action.DELETE,
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(bmID1, bmID1, bmID2)),
                asList(ExpectedResult.errors(new Notification(9800,
                                Api5ErrorDetails.BID_MODIFIER_COULD_NOT_BE_PRESENTED_MORE_THAN_ONCE)),
                        ExpectedResult.errors(new Notification(9800,
                                Api5ErrorDetails.BID_MODIFIER_COULD_NOT_BE_PRESENTED_MORE_THAN_ONCE)),
                        ExpectedResult.success(bmID2)),
                asList((ExpectedResult) ExpectedResult.errors(new Notification(9800,
                                Api5ErrorDetails.BID_MODIFIER_SHOULD_NOT_BE_PRESENTED_TWICE)).withId(bmID1),
                        (ExpectedResult) ExpectedResult.errors(new Notification(9800,
                                Api5ErrorDetails.BID_MODIFIER_SHOULD_NOT_BE_PRESENTED_TWICE)).withId(bmID1),
                        ExpectedResult.success(bmID2))
        );

        List<BidModifierGetItem> bms = api.userSteps.bidModifiersSteps().bidModifiersGetById(bmID1, bmID2);
        List<Long> bmIds = extract(bms, on(BidModifierGetItem.class).getId());
        assertThat("корректировка спровоцировавшая ошибку не удалена", bmIds, contains(equalTo(bmID1)));
    }
}
