package ru.yandex.autotests.direct.api.bidmodifiers.delete;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.GeoMultiplierValuesRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.HierarchicalMultipliersRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.model.RegionIDValues;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.RegionalAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.TestSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.directapi.model.api5.bidmodifiers.RegionalAdjustmentMap.REGIONAL_ADJUSTMENT_MAX;
import static ru.yandex.autotests.directapi.model.api5.bidmodifiers.RegionalAdjustmentMap.REGIONAL_ADJUSTMENT_MIN;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(BidModifiersFeatures.DELETE)
@RunWith(Parameterized.class)
@Issue("https://st.yandex-team.ru/DIRECT-67328")
@Description("Проверка удаления скрытой корректировки для Крыма")
public class DeleteHiddenBidModifiersForCrimeaTest {

    private static final Integer modifierNotHidden = 0;
    private static final Integer modifierIsHidden = 1;
    private static final Long crimeaGeoId = Long.valueOf(RegionIDValues.CRIMEA.getId());

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    @Parameterized.Parameter()
    public String description;

    @Parameterized.Parameter(1)
    public String client;

    @Parameterized.Parameter(2)
    public Long regionId;

    private Long campaignId;
    private Long geoModifierId;
    private Long hierarchicalMultiplierId;

    private DirectJooqDbSteps jooqSteps;

    @Parameterized.Parameters(name = "{0}")
    public static Collection testData() {
        Object[][] data = new Object[][]{
                {"Клиент из России", BidModifiersLogins.SINGLE_CLIENT_ADD, Long.valueOf(RegionIDValues.RUSSIA.getId())},
                {"Клиент из Украины", Logins.LOGIN_FOR_UAH, Long.valueOf(RegionIDValues.UKRAINE.getId())},
        };
        return Arrays.asList(data);
    }

    @Before
    public void prepare() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);

        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(client);

        List<Long> bmIds = api.as(client).userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withRegionalAdjustment(new RegionalAdjustmentMap().withBidModifier(REGIONAL_ADJUSTMENT_MIN)
                        .withRegionId(regionId)).withCampaignId(campaignId));
        assumeThat("добавлена 1 гео корректировка", bmIds, hasSize(1));

        geoModifierId = bmIds.get(0);

        jooqSteps = api.userSteps.getDirectJooqDbSteps();
        jooqSteps.useShardForLogin(client);

        List<HierarchicalMultipliersRecord> multipliers =
                jooqSteps.multipliersSteps().getHierarchicalMultipliersByCid(campaignId);
        assumeThat("в бд 1 иерархическая корректировка", multipliers, hasSize(1));

        hierarchicalMultiplierId = multipliers.get(0).getHierarchicalMultiplierId().longValue();

        List<GeoMultiplierValuesRecord> geoMultipliers = jooqSteps.multipliersSteps()
                .getGeoMultiplierValuesByHierarchicalMultipliersId(hierarchicalMultiplierId);
        assumeThat("в бд 2 гео корректировки", geoMultipliers, hasSize(2));
        assumeThat("мультипликатор корректировки для Крыма ожидаемый",
                geoMultipliers.stream().filter(e -> e.getRegionId().equals(crimeaGeoId))
                        .map(e -> Integer.valueOf(e.getMultiplierPct())).collect(Collectors.toList()),
                contains(equalTo(REGIONAL_ADJUSTMENT_MIN)));
        assumeThat("скрытая корректировка для Крыма",
                geoMultipliers.stream().filter(e -> e.getRegionId().equals(crimeaGeoId))
                        .map(GeoMultiplierValuesRecord::getIsHidden).collect(Collectors.toList()),
                contains(equalTo(modifierIsHidden)));
    }

    @Test
    @Description("При удалении корректировки на родительский регион скрытая корректировка на Крым тоже должна удалиться")
    public void testRemoveHiddenModifierForCrimea() {
        api.as(client).userSteps.bidModifiersSteps().shouldGetResultOn(Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(geoModifierId)),
                ExpectedResult.success(geoModifierId));

        List<HierarchicalMultipliersRecord> multipliers =
                jooqSteps.multipliersSteps().getHierarchicalMultipliersByCid(campaignId);
        assumeThat("иерархическая корректировка удалена", multipliers, hasSize(0));

        List<GeoMultiplierValuesRecord> geoMultipliers = jooqSteps.multipliersSteps()
                .getGeoMultiplierValuesByHierarchicalMultipliersId(hierarchicalMultiplierId);
        assertThat("гео корректировки удалены", geoMultipliers, hasSize(0));
    }

    @Test
    @Description("Если есть явная корректировка на Крым, то при удалении корректировки на родительский регион корректировка на Крым не должна удалиться")
    public void testKeepModifierForCrimea() {
        List<Long> bmIds = api.as(client).userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withRegionalAdjustment(new RegionalAdjustmentMap().withBidModifier(REGIONAL_ADJUSTMENT_MAX)
                        .withRegionId(crimeaGeoId)).withCampaignId(campaignId));
        assumeThat("добавлена корректировка ставки", bmIds, hasSize(1));

        List<HierarchicalMultipliersRecord> multipliers =
                jooqSteps.multipliersSteps().getHierarchicalMultipliersByCid(campaignId);
        assumeThat("в бд 1 иерархическая корректировка", multipliers, hasSize(1));

        Long multiplierId = multipliers.get(0).getHierarchicalMultiplierId().longValue();

        List<GeoMultiplierValuesRecord> geoMultipliers = jooqSteps.multipliersSteps()
                .getGeoMultiplierValuesByHierarchicalMultipliersId(multiplierId);
        assumeThat("в бд 2 гео корректировки", geoMultipliers, hasSize(2));
        assumeThat("мультипликатор корректировки для Крыма ожидаемый",
                geoMultipliers.stream().filter(e -> e.getRegionId().equals(crimeaGeoId))
                        .map(e -> Integer.valueOf(e.getMultiplierPct())).collect(Collectors.toList()),
                contains(equalTo(REGIONAL_ADJUSTMENT_MAX)));
        TestSteps.assertThat("корректировка для Крыма не скрытая",
                geoMultipliers.stream().filter(e -> e.getRegionId().equals(crimeaGeoId))
                        .map(GeoMultiplierValuesRecord::getIsHidden).collect(Collectors.toList()),
                contains(equalTo(modifierNotHidden)));

        api.as(client).userSteps.bidModifiersSteps().shouldGetResultOn(Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(geoModifierId)),
                ExpectedResult.success(geoModifierId));

        multipliers = jooqSteps.multipliersSteps().getHierarchicalMultipliersByCid(campaignId);
        assumeThat("в бд 1 иерархическая корректировка", multipliers, hasSize(1));

        geoMultipliers = jooqSteps.multipliersSteps()
                .getGeoMultiplierValuesByHierarchicalMultipliersId(hierarchicalMultiplierId);
        assumeThat("в бд 1 гео корректировки", geoMultipliers, hasSize(1));
        assertThat("в бд есть корректировка для Крыма",
                geoMultipliers.stream().filter(e -> e.getRegionId().equals(crimeaGeoId))
                        .collect(Collectors.toList()),
                hasSize(1));
    }
}
