package ru.yandex.autotests.direct.api.bidmodifiers.delete;

import java.util.List;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.bidmodifiers.BidModifierFieldEnum;
import com.yandex.direct.api.v5.bidmodifiers.BidModifierGetItem;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifiersSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.IncomeGradeAdjustmentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(BidModifiersFeatures.DELETE)
@Description("Удаление корректировки ставок на доход для кампании типа ДТО")
public class DeleteIncomeGradeBidModifiersTest {

    private static final String LOGIN = BidModifiersLogins.SINGLE_CLIENT_DELETE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private Long groupId;
    private Long campaignId;

    @Before
    @Step("Подготовка данных для теста")
    public void prepareGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        campaignId = api.as(LOGIN).userSteps.campaignSteps().addDefaultDynamicTextCampaign();
        groupId = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(campaignId);
    }

    @Test
    public void deleteIncomeGradeBidModifierForAdGroup() {
        List<Long> addedBidModifiers = api.userSteps.bidModifiersSteps().bidModifiersAdd(
                new BidModifierAddMap()
                        .withAdGroupId(groupId)
                        .withIncomeGradeAdjustments(
                                new IncomeGradeAdjustmentMap()
                                        .withDefaultBidModifier()
                                        .withDefaultGrade()
                        )
        );
        assumeThat("корректировка ставки установлена", addedBidModifiers, hasSize(1));

        api.userSteps.bidModifiersSteps().bidModifiersDelete(addedBidModifiers.stream().toArray(Long[]::new));

        List<BidModifierGetItem> result = api.userSteps.bidModifiersSteps().bidModifiersGetByAdGroupId(groupId);
        assumeThat("не найдена ни одна корректировка", result, hasSize(0));
    }

    @Test
    public void deleteIncomeGradeBidModifierForCampaign() {
        List<Long> addedBidModifiers = api.userSteps.bidModifiersSteps().bidModifiersAdd(
                new BidModifierAddMap()
                        .withCampaignId(campaignId)
                        .withIncomeGradeAdjustments(
                                new IncomeGradeAdjustmentMap()
                                        .withDefaultBidModifier()
                                        .withDefaultGrade()
                        )
        );
        assumeThat("корректировка ставки установлена", addedBidModifiers, hasSize(1));

        api.userSteps.bidModifiersSteps().bidModifiersDelete(addedBidModifiers.stream().toArray(Long[]::new));

        List<BidModifierGetItem> result = api.userSteps.bidModifiersSteps().bidModifiersGetByAdGroupId(campaignId);
        assumeThat("не найдена ни одна корректировка", result, hasSize(0));
    }

    @Test
    public void deleteIncomeGradeBidModifierBySelectedIds() {
        List<Long> addedBidModifiers = api.userSteps.bidModifiersSteps().bidModifiersAdd(
                new BidModifierAddMap()
                        .withCampaignId(campaignId)
                        .withIncomeGradeAdjustments(
                                new IncomeGradeAdjustmentMap()
                                        .withDefaultBidModifier()
                                        .withDefaultGrade()
                        )
        );
        assumeThat("корректировка ставки установлена", addedBidModifiers, hasSize(1));

        List<BidModifierGetItem> actualBidModifiers = api.userSteps.bidModifiersSteps().bidModifiersGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new BidModifiersSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                                        .withAllLevels()
                        )
                        .withFieldNames(BidModifierFieldEnum.ID)
        );

        List<Long> actualBidModifierIds =
                actualBidModifiers.stream().map(BidModifierGetItem::getId).collect(Collectors.toList());

        api.userSteps.bidModifiersSteps().bidModifiersDelete(actualBidModifierIds.stream().toArray(Long[]::new));

        List<BidModifierGetItem> result = api.userSteps.bidModifiersSteps().bidModifiersGetByAdGroupId(groupId);
        assumeThat("не найдена ни одна корректировка", result, hasSize(0));
    }

}
