package ru.yandex.autotests.direct.api.bidmodifiers.get;


import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import com.yandex.direct.api.v5.bidmodifiers.BidModifierFieldEnum;
import com.yandex.direct.api.v5.bidmodifiers.BidModifierGetItem;
import com.yandex.direct.api.v5.bidmodifiers.BidModifierLevelEnum;
import org.apache.commons.lang3.StringUtils;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.utils.ReflectionUtils;
import ru.yandex.autotests.direct.utils.converter.EnumToStringValueConverter;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifiersSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.convert;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 14.07.15.
 */
@Aqua.Test
@Features(BidModifiersFeatures.GET)
@Tag(TagDictionary.TRUNK)
@Description("Проверка выдачи при использовании фильтра по FieldNames")
@RunWith(Parameterized.class)
public class GetBidModifiersFieldNamesTest {
    private static LogSteps log = LogSteps.getLogger(GetBidModifiersFieldNamesTest.class);
    private static final String client = BidModifiersLogins.SINGLE_CLIENT_GET;

    private static Long campaignID;
    private static long groupID;

    private static long bmID1;
    private static long bmID2;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public String criteriaName;

    @Parameterized.Parameter(value = 2)
    public BidModifiersSelectionCriteriaMap criteria;

    @Parameterized.Parameter(value = 3)
    public List<BidModifierFieldEnum> fieldNames;

    @Parameterized.Parameters(name = "criteria = {1}, test = {0}")
    public static Collection fieldsValue() {
        prepareBidModifiers();
        Object[][] data = new Object[][]{
                {"Поле Id", "Id", new BidModifiersSelectionCriteriaMap().withIds(bmID1),
                        Arrays.asList(BidModifierFieldEnum.ID)},
                {"Поле adGroupId", "Id", new BidModifiersSelectionCriteriaMap().withIds(bmID1),
                        Arrays.asList(BidModifierFieldEnum.AD_GROUP_ID)},
                {"Поле campaignId", "Id", new BidModifiersSelectionCriteriaMap().withIds(bmID1),
                        Arrays.asList(BidModifierFieldEnum.CAMPAIGN_ID)},
                {"Поле level", "Id", new BidModifiersSelectionCriteriaMap().withIds(bmID1),
                        Arrays.asList(BidModifierFieldEnum.LEVEL)},
                {"Поле type", "Id", new BidModifiersSelectionCriteriaMap().withIds(bmID1),
                        Arrays.asList(BidModifierFieldEnum.TYPE)},
                {"Повторяющиеся имена полей", "Id", new BidModifiersSelectionCriteriaMap().withIds(bmID1),
                        Arrays.asList(BidModifierFieldEnum.ID, BidModifierFieldEnum.ID)},
                {"Все поля", "Id", new BidModifiersSelectionCriteriaMap().withIds(bmID1),
                        Arrays.asList(BidModifierFieldEnum.ID,
                                BidModifierFieldEnum.AD_GROUP_ID,
                                BidModifierFieldEnum.CAMPAIGN_ID,
                                BidModifierFieldEnum.LEVEL,
                                BidModifierFieldEnum.TYPE)},

                {"Все поля", "adGroupId", new BidModifiersSelectionCriteriaMap().withAdGroupIds(groupID),
                        Arrays.asList(BidModifierFieldEnum.ID,
                                BidModifierFieldEnum.AD_GROUP_ID,
                                BidModifierFieldEnum.CAMPAIGN_ID,
                                BidModifierFieldEnum.LEVEL,
                                BidModifierFieldEnum.TYPE)},

                {"Все поля", "campaignId", new BidModifiersSelectionCriteriaMap().withCampaignIds(campaignID),
                        Arrays.asList(BidModifierFieldEnum.ID,
                                BidModifierFieldEnum.AD_GROUP_ID,
                                BidModifierFieldEnum.CAMPAIGN_ID,
                                BidModifierFieldEnum.LEVEL,
                                BidModifierFieldEnum.TYPE)}
        };
        return Arrays.asList(data);
    }

    private static void prepareBidModifiers() {
        log.info("Подготовим корректировки ставок");
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);

        bmID1 = api.userSteps.bidModifiersSteps().addBidModifierMobileToGroup(groupID);
        bmID2 = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID);
    }

    @Test
    public void filedNamesTest() {
        List<BidModifierGetItem> bms = api.userSteps.bidModifiersSteps().bidModifiersGet(
                new GetRequestMap()
                        .withSelectionCriteria(criteria
                                .withLevels(BidModifierLevelEnum.AD_GROUP))
                        .withFieldNames((BidModifierFieldEnum[]) fieldNames.toArray()));
        assumeThat("получены корректировки ставок", bms, hasSize(greaterThan(0)));

        log.info("Проверим набор корректировок ставок на наличие запрошенных полей");
        for (BidModifierGetItem item : bms) {
            checkFieldNames(item, fieldNames);
        }
    }

    private void checkFieldNames(BidModifierGetItem item, List<BidModifierFieldEnum> fields) {
        List<String> gotFileds = getNonNullFieldsNames(item);

        //exclude field duplicates
        HashSet<BidModifierFieldEnum> fieldsSet = new HashSet<>(fields);
        List<BidModifierFieldEnum> nonDuplicatedFields = new ArrayList<>(fieldsSet);

        String[] expectedFields = convert(new ArrayList<>(nonDuplicatedFields), new EnumToStringValueConverter())
                .toArray(new String[nonDuplicatedFields.size()]);

        assertThat("вернулся объект с корректным набором запрошенных полей", gotFileds, containsInAnyOrder(expectedFields));
    }

    private List<String> getNonNullFieldsNames(Object object) {
        if (object == null) {
            throw new IllegalArgumentException("Необходимо передать объект для извлечения имен инициализированных полей");
        }
        log.info("Извлечем имена инициализированных полей объекта");
        List<String> nonNullFileds = new ArrayList<>();
        for (Field field : object.getClass().getDeclaredFields()) {
            if (ReflectionUtils.invokeGetter(object, StringUtils.capitalize(field.getName())) != null) {
                nonNullFileds.add(StringUtils.capitalize(field.getName()));
            }
        }
        return nonNullFileds;
    }

    @AfterClass
    public static void putCampaignToRemove(){
        //workaround для корректроного создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().putCampaignToQueue(client, campaignID);
    }
}
