package ru.yandex.autotests.direct.api.bidmodifiers.get;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.bidmodifiers.BidModifierLevelEnum;
import com.yandex.direct.api.v5.general.AgeRangeEnum;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifiersSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.DemographicsAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.LimitOffsetMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

/**
 * Created by chicos on 09.07.15.
 */
@Aqua.Test
@Features(BidModifiersFeatures.GET)
@Description("Проверка негативных сценариев для полей Limit и Offset")
@RunWith(Parameterized.class)
public class GetBidModifiersLimitOffsetNegativeTest {
    private static LogSteps log = LogSteps.getLogger(GetBidModifiersLimitOffsetNegativeTest.class);
    private static final String client = BidModifiersLogins.SINGLE_CLIENT_GET;

    private static Long campaignID;
    private static long groupID;

    private static long bmID1;
    private static long bmID2;
    private static long bmID3;
    private static long bmID4;
    private static long bmID5;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(value = 0)
    public Long limit;

    @Parameterized.Parameter(value = 1)
    public Long offset;

    @Parameterized.Parameter(value = 2)
    public String description;

    @Parameterized.Parameter(value = 3)
    public BidModifiersSelectionCriteriaMap selectionCriteriaMap;

    @Parameterized.Parameter(value = 4)
    public Api5Error expectedError;

    @Parameterized.Parameters(name = "criteria = {2} limit = {0}, offset = {1}")
    public static Collection limitsAndOffsets() {
        prepareBidModifiers();
        Long[] defaultBidModifierIds = new Long[]{bmID1, bmID2, bmID3, bmID4, bmID5};
        Object[][] data = new Object[][]{
                //запрос по ID
                {0L, 2L, "ID",
                        new BidModifiersSelectionCriteriaMap().withIds(defaultBidModifierIds),
                        new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0)},
                {0L, null, "ID",
                        new BidModifiersSelectionCriteriaMap().withIds(defaultBidModifierIds),
                        new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0)},
                {-2l, null, "ID",
                        new BidModifiersSelectionCriteriaMap().withIds(defaultBidModifierIds),
                        new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0)},
                {-6l, null, "ID",
                        new BidModifiersSelectionCriteriaMap().withIds(defaultBidModifierIds),
                        new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0)},
                {-5l, null, "ID",
                        new BidModifiersSelectionCriteriaMap().withIds(defaultBidModifierIds),
                        new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0)},
                {null, -1l, "ID",
                        new BidModifiersSelectionCriteriaMap().withIds(defaultBidModifierIds),
                        new Api5Error(4002, Api5ErrorDetails.OFFSET_MUST_BE_MORE_OR_EQUAL_THAN_0)},
                {null, -6l, "ID",
                        new BidModifiersSelectionCriteriaMap().withIds(defaultBidModifierIds),
                        new Api5Error(4002, Api5ErrorDetails.OFFSET_MUST_BE_MORE_OR_EQUAL_THAN_0)},
                {2l, -3l, "ID",
                        new BidModifiersSelectionCriteriaMap().withIds(defaultBidModifierIds),
                        new Api5Error(4002, Api5ErrorDetails.OFFSET_MUST_BE_MORE_OR_EQUAL_THAN_0)},
                {-2l, -1l, "ID",
                        new BidModifiersSelectionCriteriaMap().withIds(defaultBidModifierIds),
                        new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0)},
                //DIRECT-41922
                {10001l, null, "ID",
                        new BidModifiersSelectionCriteriaMap().withIds(defaultBidModifierIds),
                        new Api5Error(4002, Api5ErrorDetails.LIMIT_CANNOT_BE_MORE, LimitOffsetMap.LIMIT_MAX_LENGTH)},

                //DIRECT-45928
                {10000l, 110001l, "ID",
                        new BidModifiersSelectionCriteriaMap().withIds(defaultBidModifierIds),
                        new Api5Error(9301
                                , Api5ErrorDetails.LIMIT_AND_OFFSET_CANNOT_BE_MORE
                                , LimitOffsetMap.LIMIT_OFFSET_MAX_LENGTH
                        )},
                {1l, 120000l, "ID",
                        new BidModifiersSelectionCriteriaMap().withIds(defaultBidModifierIds),
                        new Api5Error(9301
                                , Api5ErrorDetails.LIMIT_AND_OFFSET_CANNOT_BE_MORE
                                , LimitOffsetMap.LIMIT_OFFSET_MAX_LENGTH
                        )},
                {null, 110001l, "ID",
                        new BidModifiersSelectionCriteriaMap().withIds(defaultBidModifierIds),
                        new Api5Error(9301
                                , Api5ErrorDetails.LIMIT_AND_OFFSET_CANNOT_BE_MORE
                                , LimitOffsetMap.LIMIT_OFFSET_MAX_LENGTH
                        )},

                //запрос по groupID
                {-5l, null, "groupID",
                        new BidModifiersSelectionCriteriaMap().withAdGroupIds(groupID),
                        new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0)},
                {2l, -3l, "groupID",
                        new BidModifiersSelectionCriteriaMap().withAdGroupIds(groupID),
                        new Api5Error(4002, Api5ErrorDetails.OFFSET_MUST_BE_MORE_OR_EQUAL_THAN_0)},

                //запрос по campaignID
                {-5l, null, "campaignID",
                        new BidModifiersSelectionCriteriaMap().withAdGroupIds(groupID),
                        new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0)},
                {2l, -3l, "campaignID",
                        new BidModifiersSelectionCriteriaMap().withAdGroupIds(groupID),
                        new Api5Error(4002, Api5ErrorDetails.OFFSET_MUST_BE_MORE_OR_EQUAL_THAN_0)},
        };
        return Arrays.asList(data);
    }

    private static void prepareBidModifiers() {
        log.info("Подготовим корректировки ставок");
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        Integer conditionID = api.userSteps.retargetingSteps().addRandomRetargetingCondition(client);

        bmID1 = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID);
        bmID2 = api.userSteps.bidModifiersSteps().addBidModifierMobileToGroup(groupID);
        bmID3 = api.userSteps.bidModifiersSteps().addBidModifierRetargetingToGroup(conditionID.longValue(), groupID);
        bmID4 = api.userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                        .withAge(AgeRangeEnum.AGE_45)
                        .withBidModifier(120))
                .withAdGroupId(groupID)).get(0);
        bmID5 = api.userSteps.bidModifiersSteps().bidModifiersAdd(new BidModifierAddMap()
                .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                        .withAge(AgeRangeEnum.AGE_35_44)
                        .withBidModifier(130))
                .withAdGroupId(groupID)).get(0);
    }

    @Before
    public void clearUnits(){
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
    }

    @Test
    public void limitOffsetTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(selectionCriteriaMap
                                .withLevels(BidModifierLevelEnum.AD_GROUP))
                        .withPage(new LimitOffsetMap()
                                .withLimit(limit)
                                .withOffset(offset))
                        .withAllFieldNames(),
                expectedError);
    }

    @AfterClass
    public static void putCampaignToRemove(){
        //workaround для корректроного создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().putCampaignToQueue(client, campaignID);
    }
}
