package ru.yandex.autotests.direct.api.bidmodifiers.get;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import com.yandex.direct.api.v5.bidmodifiers.BidModifierGetItem;
import org.hamcrest.Matcher;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifiersSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.extract;
import static ch.lambdaj.Lambda.on;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.emptyIterable;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by chicos on 27.07.15.
 */
@Aqua.Test
@Features(BidModifiersFeatures.GET)
@RunWith(Parameterized.class)
@Description("Проверка негативных сценариев выборки корректировок ставок по параметрам SelectionCriteria")
public class GetBidModifiersSelectionCriteriaNegativeTest {
    private static LogSteps log = LogSteps.getLogger(GetBidModifiersSelectionCriteriaNegativeTest.class);

    private static String client = BidModifiersLogins.SINGLE_CLIENT_GET;
    private static String anotherClient = BidModifiersLogins.SINGLE_CLIENT_ANOTHER;

    private static Long bmID;
    private static Long bidModifierDeleted;

    private static Long bmOfDeletedGroup;
    private static Long bmOfDeletedCampaign;
    private static Long bmOfArchivedCampaign;
    private static Long bmOfArchivedGroup;

    private static Long campaignID;
    private static Long campaignDeleted;
    private static Long campaignArchived;

    private static Long groupID;
    private static Long groupDeleted;
    private static Long groupArchived;

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public String operator;

    @Parameterized.Parameter(value = 2)
    public BidModifiersSelectionCriteriaMap criteriaMap;

    @Parameterized.Parameter(value = 3)
    public Matcher expectedResult;

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection dataSet() {
        prepareBidModifiers();
        log.info("BidModifiers ready!");
        Object[][] data = new Object[][]{
                {"Некорректный adGroupId",
                        client, new BidModifiersSelectionCriteriaMap().withAdGroupIds(0l).withAllLevels(),
                        emptyIterable()},
                {"Некорректный campaignId",
                        client, new BidModifiersSelectionCriteriaMap().withCampaignIds(0l).withAllLevels(),
                        emptyIterable()},
                {"Удаленная корректировка ставок",
                        client, new BidModifiersSelectionCriteriaMap().withIds(bidModifierDeleted).withAllLevels(),
                        emptyIterable()},
                {"Несуществующая корректировка ставок",
                        client, new BidModifiersSelectionCriteriaMap().withIds(38497248L).withAllLevels(),
                        emptyIterable()},
                {"Некорректный Id корректировки ставок",
                        client, new BidModifiersSelectionCriteriaMap().withIds(0l).withAllLevels(),
                        emptyIterable()},
                {"Корректный и некорректный Id корректировки ставок",
                        client, new BidModifiersSelectionCriteriaMap().withIds(bmID, -123l).withAllLevels(),
                        hasSize(1)},

                {"Корректировка ставок другого пользователя по Id",
                        anotherClient, new BidModifiersSelectionCriteriaMap().withIds(bmID).withAllLevels(),
                        emptyIterable()},
                {"Корректировка ставок другого пользователя по campaignId",
                        anotherClient, new BidModifiersSelectionCriteriaMap().withCampaignIds(campaignID).withAllLevels(),
                        emptyIterable()},
                {"Корректировка ставок другого пользователя по adGroupId",
                        anotherClient, new BidModifiersSelectionCriteriaMap().withAdGroupIds(groupID).withAllLevels(),
                        emptyIterable()},

                {"Корректировка ставок удаленной группы по Id",
                        client, new BidModifiersSelectionCriteriaMap().withIds(bmOfDeletedGroup).withAllLevels(),
                        emptyIterable()},
                {"Корректировка ставок удаленной кампании по Id",
                        client, new BidModifiersSelectionCriteriaMap().withIds(bmOfDeletedCampaign).withAllLevels(),
                        emptyIterable()},
                {"Корректировка ставок удаленной группы по groupId",
                        client, new BidModifiersSelectionCriteriaMap().withAdGroupIds(groupDeleted).withAllLevels(),
                        emptyIterable()},
                {"Корректировка ставок удаленной кампании по campaignId",
                        client, new BidModifiersSelectionCriteriaMap().withCampaignIds(campaignDeleted).withAllLevels(),
                        emptyIterable()},

                {"Корректировка ставок архивной кампании по campaignId",
                        client, new BidModifiersSelectionCriteriaMap().withCampaignIds(campaignArchived).withAllLevels(),
                        containsInAnyOrder(bmOfArchivedCampaign, bmOfArchivedGroup)},
                {"Корректировка ставок архивной кампании по Id",
                        client, new BidModifiersSelectionCriteriaMap().withIds(bmOfArchivedCampaign).withAllLevels(),
                        containsInAnyOrder(bmOfArchivedCampaign)},
                {"Корректировка ставок архивной группы по groupId",
                        client, new BidModifiersSelectionCriteriaMap().withAdGroupIds(groupArchived).withAllLevels(),
                        containsInAnyOrder(bmOfArchivedGroup)},

        };
        return Arrays.asList(data);
    }

    @Test
    public void negativeCriteriaTest() {
        List<BidModifierGetItem> bms = api.as(operator).userSteps.bidModifiersSteps().bidModifiersGet(new GetRequestMap()
                .withSelectionCriteria(criteriaMap).withAllFieldNames());
        List<Long> bmIds = extract(bms, on(BidModifierGetItem.class).getId());
        assertThat("вернулся корректный набор корректировок ставок", bmIds, expectedResult);
    }

    @AfterClass
    public static void putCampaignToRemove(){
        //workaround для корректроного создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().putCampaignToQueue(client, campaignID);
    }

    private static void prepareBidModifiers(){
        api.as(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        //workaround для создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().removeCampaignsFromQueue(Arrays.asList(campaignID));

        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        bmID = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID);

        bidModifierDeleted = api.userSteps.bidModifiersSteps().addBidModifierMobileToGroup(groupID);
        api.userSteps.bidModifiersSteps().bidModifiersDelete(bidModifierDeleted);

        campaignDeleted = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        groupDeleted = api.userSteps.adGroupsSteps().addDefaultGroup(campaignDeleted);
        bmOfDeletedGroup = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupDeleted);
        bmOfDeletedCampaign = api.userSteps.bidModifiersSteps().addBidModifierMobileToCampaign(campaignDeleted);
        api.userSteps.campaignSteps().campaignsDelete(campaignDeleted);

        campaignArchived = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        groupArchived = api.userSteps.adGroupsSteps().addDefaultGroup(campaignArchived);
        bmOfArchivedCampaign = api.userSteps.bidModifiersSteps().addBidModifierMobileToCampaign(campaignArchived);
        bmOfArchivedGroup = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupArchived);
        api.userSteps.campaignSteps().campaignsSuspend(campaignArchived);
        api.userSteps.campaignSteps().campaignsArchive(campaignArchived);
    }
}
