package ru.yandex.autotests.direct.api.bidmodifiers.get;


import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import com.yandex.direct.api.v5.bidmodifiers.BidModifierGetItem;
import com.yandex.direct.api.v5.bidmodifiers.BidModifierLevelEnum;
import com.yandex.direct.api.v5.bidmodifiers.BidModifierTypeEnum;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifiersSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.extract;
import static ch.lambdaj.Lambda.on;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by chicos on 27.07.15
 */
@Aqua.Test
@Features(BidModifiersFeatures.GET)
@Description("Проверка комбинаций параметров в SelectionCriteria при запросе корректировок ставок")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class GetBidModifiersSelectionCriteriaTest {
    private static LogSteps log = LogSteps.getLogger(GetBidModifiersSelectionCriteriaTest.class);
    private static final String client = BidModifiersLogins.SINGLE_CLIENT_GET;

    //кампании
    private static Long campaignID1;
    private static Long campaignID2;
    private static Long campaignID3;

    //группы 1-й кампании
    private static Long groupID11;

    //группы 2-й кампании
    private static Long groupID22;
    private static Long groupID23;

    //группы 3-й кампании
    private static Long groupID34;

    //корректировки 1-й кампании
    private static Long bmCampaignID11;
    private static Long bmCampaignID12;

    //корректировки 2-й кампании
    private static Long bmCampaignID21;

    //корректировки 3-й кампании
    private static Long bmCampaignID31;
    private static Long bmCampaignID32;

    //корректировки 1-й группы
    private static Long bmGroupID11;
    private static Long bmGroupID12;

    //корректировки 2-й группы
    private static Long bmGroupID21;

    //корректировки 3-й группы
    private static Long bmGroupID31;

    //корректировки 4-й группы
    private static Long bmGroupID41;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public Long[] campaignIds;

    @Parameterized.Parameter(value = 2)
    public Long[] groupIds;

    @Parameterized.Parameter(value = 3)
    public Long[] bmIds;

    @Parameterized.Parameter(value = 4)
    public BidModifierTypeEnum[] types;

    @Parameterized.Parameter(value = 5)
    public BidModifierLevelEnum[] levels;

    @Parameterized.Parameter(value = 6)
    public List<Long> expectedBidModifiers;

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection fieldValues() {
        prepareBidModifiers();
        log.info("BidModifiers ready!");
        Object[][] data = new Object[][]{
                {"ID одной кампании",
                        new Long[]{campaignID2}, null, null, null, getAllLevels(),
                        Arrays.asList(bmCampaignID21, bmGroupID21, bmGroupID31)},
                {"ID двух кампаний",
                        new Long[]{campaignID2, campaignID3}, null, null, null, getAllLevels(),
                        Arrays.asList(bmCampaignID21, bmGroupID21, bmGroupID31, bmCampaignID31, bmCampaignID32, bmGroupID41)},
                {"ID кампании с несколькими корректировками",
                        new Long[]{campaignID1}, null, null, null, getAllLevels(),
                        Arrays.asList(bmCampaignID11, bmCampaignID12, bmGroupID11, bmGroupID12)},
                {"ID одной группы",
                        null, new Long[]{groupID22}, null, null, getAllLevels(),
                        Arrays.asList(bmGroupID21)},
                {"ID двух групп",
                        null, new Long[]{groupID23, groupID34}, null, null, getAllLevels(),
                        Arrays.asList(bmGroupID31, bmGroupID41)},
                {"ID группы с двумя корректировками",
                        null, new Long[]{groupID11}, null, null, getAllLevels(),
                        Arrays.asList(bmGroupID11, bmGroupID12)},
                {"ID одной корректировки",
                        null, null, new Long[]{bmGroupID31}, null, getAllLevels(),
                        Arrays.asList(bmGroupID31)},
                {"ID двух корректировок",
                        null, null, new Long[]{bmGroupID21, bmGroupID41}, null, getAllLevels(),
                        Arrays.asList(bmGroupID21, bmGroupID41)},
                {"ID кампании и группы",
                        new Long[]{campaignID2}, new Long[]{groupID23}, null, null, getAllLevels(),
                        Arrays.asList(bmGroupID31)},
                {"ID кампании и чужой группы",
                        new Long[]{campaignID3}, new Long[]{groupID22}, null, null, getAllLevels(),
                        Arrays.asList()},
                {"ID кампании и ID корректировки группы",
                        new Long[]{campaignID2}, null, new Long[]{bmGroupID21}, null, getAllLevels(),
                        Arrays.asList(bmGroupID21)},
                {"ID кампании и ID корректировки кампании",
                        new Long[]{campaignID1}, null, new Long[]{bmCampaignID12}, null, getAllLevels(),
                        Arrays.asList(bmCampaignID12)},
                {"ID группы и ID корректировки группы",
                        null, new Long[]{groupID11}, new Long[]{bmGroupID12}, null, getAllLevels(),
                        Arrays.asList(bmGroupID12)},
                {"ID группы и ID корректировки чужой группы",
                        null, new Long[]{groupID22}, new Long[]{bmGroupID31}, null, getAllLevels(),
                        Arrays.asList()},
                {"ID кампании и ID корректировки чужой группы",
                        new Long[]{campaignID3}, null, new Long[]{bmGroupID31}, null, getAllLevels(),
                        Arrays.asList()},
                {"ID кампании и ID корректировки чужой кампании",
                        new Long[]{campaignID2}, null, new Long[]{bmCampaignID31}, null, getAllLevels(),
                        Arrays.asList()},
                {"ID кампании, группы и корректировки группы",
                        new Long[]{campaignID1}, new Long[]{groupID11}, new Long[]{bmGroupID12}, null, getAllLevels(),
                        Arrays.asList(bmGroupID12)},
                {"ID кампании, группы и корректировки кампании",
                        new Long[]{campaignID1}, new Long[]{groupID11}, new Long[]{bmCampaignID11}, null, getAllLevels(),
                        Arrays.asList()},
                {"ID кампании, группы и чужой корректировки группы",
                        new Long[]{campaignID2}, new Long[]{groupID23}, new Long[]{bmGroupID21}, null, getAllLevels(),
                        Arrays.asList()},
                {"ID корректировки группы и уровень кампании",
                        null, null, new Long[]{bmGroupID41}, null, new BidModifierLevelEnum[]{BidModifierLevelEnum.CAMPAIGN},
                        Arrays.asList()},
                {"ID корректировки и другой тип корректировки",
                        null, null, new Long[]{bmGroupID31}, new BidModifierTypeEnum[]{BidModifierTypeEnum.RETARGETING_ADJUSTMENT}, getAllLevels(),
                        Arrays.asList()},
                {"ID кампании и уровень кампании",
                        new Long[]{campaignID1}, null, null, null, new BidModifierLevelEnum[]{BidModifierLevelEnum.CAMPAIGN},
                        Arrays.asList(bmCampaignID11, bmCampaignID12)},
                {"ID двух кампаний и уровень кампании",
                        new Long[]{campaignID3, campaignID2}, null, null, null, new BidModifierLevelEnum[]{BidModifierLevelEnum.CAMPAIGN},
                        Arrays.asList(bmCampaignID21, bmCampaignID31, bmCampaignID32)},
                {"ID кампании и уровень группы",
                        new Long[]{campaignID2}, null, null, null, new BidModifierLevelEnum[]{BidModifierLevelEnum.AD_GROUP},
                        Arrays.asList(bmGroupID21, bmGroupID31)},
                {"ID кампании и тип корректировки",
                        new Long[]{campaignID1}, null, null, new BidModifierTypeEnum[]{BidModifierTypeEnum.DEMOGRAPHICS_ADJUSTMENT}, getAllLevels(),
                        Arrays.asList(bmGroupID12, bmCampaignID12)},
                {"ID кампании и тип корректировки по региону",
                        new Long[]{campaignID3}, null, null, new BidModifierTypeEnum[]{BidModifierTypeEnum.REGIONAL_ADJUSTMENT}, getAllLevels(),
                        Arrays.asList(bmCampaignID32)},
                {"ID двух кампаний и тип корректировки",
                        new Long[]{campaignID2, campaignID3}, null, null, new BidModifierTypeEnum[]{BidModifierTypeEnum.RETARGETING_ADJUSTMENT}, getAllLevels(),
                        Arrays.asList(bmGroupID21, bmCampaignID31)},
                {"ID двух кампаний и несколько типов корректировок",
                        new Long[]{campaignID2, campaignID3}, null, null, new BidModifierTypeEnum[]{BidModifierTypeEnum.RETARGETING_ADJUSTMENT, BidModifierTypeEnum.REGIONAL_ADJUSTMENT, BidModifierTypeEnum.DEMOGRAPHICS_ADJUSTMENT}, getAllLevels(),
                        Arrays.asList(bmGroupID21, bmGroupID31, bmCampaignID31, bmCampaignID32)},
                {"ID кампании и все типы корректировок",
                        new Long[]{campaignID3}, null, null, new BidModifierTypeEnum[]{BidModifierTypeEnum.RETARGETING_ADJUSTMENT, BidModifierTypeEnum.REGIONAL_ADJUSTMENT, BidModifierTypeEnum.DEMOGRAPHICS_ADJUSTMENT, BidModifierTypeEnum.MOBILE_ADJUSTMENT}, getAllLevels(),
                        Arrays.asList(bmCampaignID31, bmCampaignID32, bmGroupID41)},
                {"ID группы и уровень группы",
                        null, new Long[]{groupID11}, null, null, new BidModifierLevelEnum[]{BidModifierLevelEnum.AD_GROUP},
                        Arrays.asList(bmGroupID11, bmGroupID12)},
                {"ID двух групп и уровень группы",
                        null, new Long[]{groupID11, groupID34}, null, null, new BidModifierLevelEnum[]{BidModifierLevelEnum.AD_GROUP},
                        Arrays.asList(bmGroupID11, bmGroupID12, bmGroupID41)},
                {"ID группы и уровень кампании",
                        null, new Long[]{groupID22}, null, null, new BidModifierLevelEnum[]{BidModifierLevelEnum.CAMPAIGN},
                        Arrays.asList()},
                {"ID группы и тип корректировки",
                        null, new Long[]{groupID11}, null, new BidModifierTypeEnum[]{BidModifierTypeEnum.MOBILE_ADJUSTMENT}, getAllLevels(),
                        Arrays.asList(bmGroupID11)},
                {"ID группы и тип корректировки - пустой ответ",
                        null, new Long[]{groupID23}, null, new BidModifierTypeEnum[]{BidModifierTypeEnum.RETARGETING_ADJUSTMENT}, getAllLevels(),
                        Arrays.asList()},
                {"ID группы и все типы корректировок",
                        null, new Long[]{groupID11}, null, new BidModifierTypeEnum[]{BidModifierTypeEnum.RETARGETING_ADJUSTMENT, BidModifierTypeEnum.DEMOGRAPHICS_ADJUSTMENT, BidModifierTypeEnum.REGIONAL_ADJUSTMENT, BidModifierTypeEnum.MOBILE_ADJUSTMENT}, getAllLevels(),
                        Arrays.asList(bmGroupID11, bmGroupID12)},
                {"Несколько ID групп и несколько типов корректировок",
                        null, new Long[]{groupID11, groupID34, groupID22}, null, new BidModifierTypeEnum[]{BidModifierTypeEnum.RETARGETING_ADJUSTMENT, BidModifierTypeEnum.DEMOGRAPHICS_ADJUSTMENT}, getAllLevels(),
                        Arrays.asList(bmGroupID12, bmGroupID21)}
        };
        return Arrays.asList(data);
    }

    private static void prepareBidModifiers() {
        log.info("Prepare campaign 1");
        campaignID1 = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        //workaround для создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().removeCampaignsFromQueue(Arrays.asList(campaignID1));

        //создадим группу для кампании 1
        groupID11 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID1);

        //создадим корректировки кампании 1
        bmCampaignID11 = api.userSteps.bidModifiersSteps().addBidModifierMobileToCampaign(campaignID1);
        bmCampaignID12 = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToCampaign(campaignID1.intValue());

        //создадим корректировки группы 1
        bmGroupID11 = api.userSteps.bidModifiersSteps().addBidModifierMobileToGroup(groupID11);
        bmGroupID12 = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID11);

        log.info("Prepare campaign 2");
        campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        //workaround для создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().removeCampaignsFromQueue(Arrays.asList(campaignID2));

        //создадим группы для кампании 2
        groupID22 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID2);
        groupID23 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID2);

        //создадим корректировку кампании 2
        bmCampaignID21 = api.userSteps.bidModifiersSteps().addBidModifierMobileToCampaign(campaignID2);

        //создадим корректировку группы 2
        Integer conditionId = api.userSteps.retargetingSteps().addRandomRetargetingCondition(client);
        bmGroupID21 = api.userSteps.bidModifiersSteps().addBidModifierRetargetingToGroup(conditionId.longValue(), groupID22);
        //создадим корректировку группы 3
        bmGroupID31 = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID23);

        log.info("Prepare campaign 3");
        campaignID3 = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        //workaround для создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().removeCampaignsFromQueue(Arrays.asList(campaignID3));

        //создадим группы для кампании 3
        groupID34 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID3);

        //создадим корректировки кампании 3
        conditionId = api.userSteps.retargetingSteps().addRandomRetargetingCondition(client);
        bmCampaignID31  = api.userSteps.bidModifiersSteps().addBidModifierRetargetingToCampaign(conditionId.longValue(), campaignID3.intValue());
        bmCampaignID32  = api.userSteps.bidModifiersSteps().addBidModifierRegionalToCampaign(campaignID3);

        //создадим корректировку группы 4
        bmGroupID41 = api.userSteps.bidModifiersSteps().addBidModifierMobileToGroup(groupID34);
    }

    @Before
    public void resetUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
    }

    @Test
    public void getBidModifierTest() {
        List<BidModifierGetItem> bms = api.userSteps.bidModifiersSteps().bidModifiersGet(
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap()
                                .withCampaignIds(campaignIds)
                                .withAdGroupIds(groupIds)
                                .withIds(bmIds)
                                .withLevels(levels)
                                .withTypes(types))
                        .withAllFieldNames());
        List<Long> bmIds = extract(bms, on(BidModifierGetItem.class).getId());

        assertThat("ответ метода совпадает с ожидаемым набором", bmIds, containsInAnyOrder(expectedBidModifiers.toArray()));
    }

    private static BidModifierLevelEnum[] getAllLevels(){
        return BidModifierLevelEnum.class.getEnumConstants();
    }

    @AfterClass
    public static void putCampaignToRemove(){
        //workaround для корректроного создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().putCampaignToQueue(client, campaignID1);
        api.userSteps.campaignSteps().putCampaignToQueue(client, campaignID2);
        api.userSteps.campaignSteps().putCampaignToQueue(client, campaignID3);
    }
}
