package ru.yandex.autotests.direct.api.bidmodifiers.get;

import java.util.List;

import com.yandex.direct.api.v5.bidmodifiers.BidModifierGetItem;
import com.yandex.direct.api.v5.bidmodifiers.BidModifierLevelEnum;
import org.apache.commons.lang3.StringUtils;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifiersSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by chicos on 29.07.15.
 */
@Aqua.Test
@Features(BidModifiersFeatures.GET)
@Description("Запросы на получение BidModifier по протоколу SOAP")
public class GetBidModifiersSoapTest {
    private static final String client = BidModifiersLogins.SINGLE_CLIENT_GET;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client).protocol(ProtocolType.SOAP);

    @Rule
    public Trashman trashman = new Trashman(api);

    private static long bmId;

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepareBidModifier() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        Long groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        bmId = api.userSteps.bidModifiersSteps().addBidModifierMobileToGroup(groupID);
    }

    @Test
    @Description("Добавление корректировки ставок с пустым набором BidModifiers")
    public void requestWithoutBidModifiersFieldTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.GET,
                new GetRequestMap(),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        StringUtils.capitalize(GetRequestMap.FIELD_NAMES)));
    }

    @Test
    public void invalidEnumValueTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap()
                                .withLevels("INVALID_VALUE")
                                .withIds(123L)),
                new Api5Error(8000, Api5ErrorDetailsJava.INCORRECT_ENUMERATE_VALUE_IN_ARRAY,
                        path(StringUtils.capitalize(GetRequestMap.SELECTION_CRITERIA),
                                StringUtils.capitalize(BidModifiersSelectionCriteriaMap.LEVELS)),
                        "CAMPAIGN, AD_GROUP"));
    }

    @Test
    public void noFieldNamesTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap()
                                .withAllLevels()
                                .withIds(123L)),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        StringUtils.capitalize(GetRequestMap.FIELD_NAMES)));
    }

    @Test
    @Description("Получение корректировки ставок по протоколу SOAP")
    public void getBidModifierPositiveTest() {
        List<BidModifierGetItem> bms = api.userSteps.bidModifiersSteps().bidModifiersGet(
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap()
                                .withIds(bmId)
                                .withLevels(BidModifierLevelEnum.AD_GROUP))
                        .withAllFieldNames());
        assertThat("получена корректировка ставок", bms, hasSize(1));
    }
}
