package ru.yandex.autotests.direct.api.bidmodifiers.get;

import java.util.Arrays;

import org.apache.commons.lang3.StringUtils;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifiersSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.directapi.model.api5.bidmodifiers.GetRequestMap.SELECTION_CRITERIA;

/**
 * Created by chicos on 21.07.15.
 */
@Aqua.Test
@Features(BidModifiersFeatures.GET)
@Description("Проверка некорректной структуры запроса Get")
public class GetBidModifiersStructureNegativeTest {
    private static final String CLIENT = BidModifiersLogins.SINGLE_CLIENT_GET;

    private static long bmID;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(CLIENT).protocol(ProtocolType.JSON);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepareBidModifier() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(CLIENT);
        long groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        bmID = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID);
    }

    @Test
    public void noSelectionCriteriaInRequestTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.GET,
                new GetRequestMap().withAllFieldNames(),
                new Api5Error(8000,
                        Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                        "params",
                        StringUtils.capitalize(GetRequestMap.SELECTION_CRITERIA)),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        StringUtils.capitalize(SELECTION_CRITERIA)));
    }

    @Test
    public void noFieldNamesTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap()
                                .withIds(bmID)
                                .withAllLevels()),
                new Api5Error(8000,
                        Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                        "params",
                        StringUtils.capitalize(GetRequestMap.FIELD_NAMES)),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        StringUtils.capitalize(GetRequestMap.FIELD_NAMES)));
    }

    @Test
    public void emptySelectionCriteriaLevelsTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap())
                        .withAllFieldNames(),
                new Api5Error(8000,
                        Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                        StringUtils.capitalize(GetRequestMap.SELECTION_CRITERIA),
                        StringUtils.capitalize(BidModifiersSelectionCriteriaMap.LEVELS)),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        path(BidModifiersSelectionCriteriaMap.LEVELS)));
    }

    @Test
    public void noIdSelectionCriteriaIdsTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap().withAllLevels())
                        .withAllFieldNames(),
                new Api5Error(4001,
                        Api5ErrorDetails.IN_STRUCTURE_MUST_BE_ONE_OF_REQUIRED_FIELD,
                        StringUtils.capitalize(GetRequestMap.SELECTION_CRITERIA),
                        BidModifiersSelectionCriteriaMap.getRequiredIdNames()),
                new Api5Error(4001,
                        Api5ErrorDetails.IN_STRUCTURE_MUST_BE_ONE_OF_REQUIRED_FIELD,
                        StringUtils.capitalize(SELECTION_CRITERIA),
                        BidModifiersSelectionCriteriaMap.getRequiredIdNames()));
    }

    @Test
    public void invalidTypeEnumTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap()
                                .withIds(bmID)
                                .withAllLevels()
                                .withTypes("INVALID_VALUE"))
                        .withAllFieldNames(),
                new Api5Error(8000,
                        Api5ErrorDetails.ELEMENT_OF_ARRAY_CONTAINS_INCORRECT_ENUMERATE_VALUE,
                        StringUtils.capitalize(BidModifiersSelectionCriteriaMap.TYPES)),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.INCORRECT_ENUMERATE_VALUE_IN_ARRAY,
                        StringUtils.capitalize(SELECTION_CRITERIA) + "." +
                                StringUtils.capitalize(BidModifiersSelectionCriteriaMap.TYPES),
                        "MOBILE_ADJUSTMENT, DESKTOP_ADJUSTMENT, TABLET_ADJUSTMENT, DESKTOP_ONLY_ADJUSTMENT," +
                                " SMART_TV_ADJUSTMENT, DEMOGRAPHICS_ADJUSTMENT, RETARGETING_ADJUSTMENT," +
                                " REGIONAL_ADJUSTMENT, VIDEO_ADJUSTMENT, SMART_AD_ADJUSTMENT," +
                                " SERP_LAYOUT_ADJUSTMENT, INCOME_GRADE_ADJUSTMENT"));
    }

    @Test
    public void maxBidModifierIdsTest() {
        Long[] fakeIds = new Long[BidModifiersSelectionCriteriaMap.IDS_MAX_LENGTH + 1];
        Arrays.fill(fakeIds, 0, BidModifiersSelectionCriteriaMap.IDS_MAX_LENGTH + 1, 123456L);

        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap()
                                .withIds(fakeIds)
                                .withAllLevels())
                        .withAllFieldNames(),
                new Api5Error(4001,
                        Api5ErrorDetails.ARRAY_CANNOT_CONTAIN_MORE_THAN,
                        StringUtils.capitalize(BidModifiersSelectionCriteriaMap.IDS),
                        BidModifiersSelectionCriteriaMap.IDS_MAX_LENGTH),
                new Api5Error(4001,
                        Api5ErrorDetails.ARRAY_CANNOT_CONTAIN_MORE_THAN,
                        StringUtils.capitalize(SELECTION_CRITERIA) + "." + StringUtils
                                .capitalize(BidModifiersSelectionCriteriaMap.IDS),
                        BidModifiersSelectionCriteriaMap.IDS_MAX_LENGTH));
    }

    @Test
    public void maxAdGroupIdsTest() {
        //DIRECT-44398
        Long[] fakeIds = new Long[BidModifiersSelectionCriteriaMap.ADGROUP_IDS_MAX_LENGTH + 1];
        Arrays.fill(fakeIds, 0, BidModifiersSelectionCriteriaMap.ADGROUP_IDS_MAX_LENGTH + 1, 123456L);

        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap()
                                .withAdGroupIds(fakeIds)
                                .withAllLevels())
                        .withAllFieldNames(),
                new Api5Error(4001,
                        Api5ErrorDetails.ARRAY_CANNOT_CONTAIN_MORE_THAN,
                        StringUtils.capitalize(BidModifiersSelectionCriteriaMap.ADGROUP_IDS),
                        BidModifiersSelectionCriteriaMap.ADGROUP_IDS_MAX_LENGTH),
                new Api5Error(4001,
                        Api5ErrorDetails.ARRAY_CANNOT_CONTAIN_MORE_THAN,
                        StringUtils.capitalize(SELECTION_CRITERIA) + "." + StringUtils
                                .capitalize(BidModifiersSelectionCriteriaMap.ADGROUP_IDS),
                        BidModifiersSelectionCriteriaMap.ADGROUP_IDS_MAX_LENGTH));
    }

    @Test
    public void maxCampaignIdsTest() {
        //DIRECT-44398
        Long[] fakeIds = new Long[BidModifiersSelectionCriteriaMap.CAMPAIGN_IDS__MAX_LENGTH + 1];
        Arrays.fill(fakeIds, 0, BidModifiersSelectionCriteriaMap.CAMPAIGN_IDS__MAX_LENGTH + 1, 123456L);

        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap()
                                .withCampaignIds(fakeIds)
                                .withAllLevels())
                        .withAllFieldNames(),
                new Api5Error(4001,
                        Api5ErrorDetails.ARRAY_CANNOT_CONTAIN_MORE_THAN,
                        StringUtils.capitalize(BidModifiersSelectionCriteriaMap.CAMPAIGN_IDS),
                        BidModifiersSelectionCriteriaMap.CAMPAIGN_IDS__MAX_LENGTH),
                new Api5Error(4001,
                        Api5ErrorDetails.ARRAY_CANNOT_CONTAIN_MORE_THAN,
                        StringUtils.capitalize(SELECTION_CRITERIA) + "." + StringUtils
                                .capitalize(BidModifiersSelectionCriteriaMap.CAMPAIGN_IDS),
                        BidModifiersSelectionCriteriaMap.CAMPAIGN_IDS__MAX_LENGTH));
    }
}
