package ru.yandex.autotests.direct.api.bidmodifiers.get;

import com.sun.xml.ws.util.StringUtils;
import com.yandex.direct.api.v5.bidmodifiers.BidModifierLevelEnum;
import com.yandex.direct.api.v5.bidmodifiers.BidModifierTypeEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifiersSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 29.07.15
 */
@Aqua.Test
@Features(BidModifiersFeatures.GET)
@Stories(ApiStories.UNITS)
@Tag(TagDictionary.TRUNK)
@Description("Списание баллов при запросе корректировок ставок")
public class GetBidModifiersUnitsTest {
    private static LogSteps log = LogSteps.getLogger(GetBidModifiersUnitsTest.class);
    private static final String singleClient = BidModifiersLogins.CLIENT_UNITS_GET;

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final int GET_COST = 1;
    private static final int ITEM_COST_ON_GET = 0;
    private static final int ITEM_COST_ON_GET_FAULT = 0;
    private static final int GENERAL_FAULT_COST = 50;

    private static Long campaignID;
    private static long groupID;

    private static long bmID1;
    private static long bmID2;

    @BeforeClass
    public static void prepareBidModifiers() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(singleClient);
        campaignID = api.as(singleClient).userSteps.campaignSteps().addDefaultTextCampaign(singleClient);
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);

        bmID1 = api.userSteps.bidModifiersSteps().addBidModifierMobileToGroup(groupID);
        bmID2 = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID);
    }

    @Before
    @Step("Сбросим использованные баллы")
    public void prepareClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(singleClient);
    }

    @Test
    @Description("Получение одиночной корректировки ставки")
    public void getSingleBidModifierTest() {
        api.as(singleClient);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.bidModifiersSteps().bidModifiersGetById(bmID1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        assertThat("стоимость получения одной корректировки ставки", unitsBefore - unitsAfter, equalTo(GET_COST + ITEM_COST_ON_GET));
    }

    @Test
    @Description("Получение корректировки ставки по AdGroupId")
    public void getBidModifierByGroupIDTest() {
        api.as(singleClient);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.bidModifiersSteps().bidModifiersGetByAdGroupId(groupID);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        assertThat("стоимость получения одной корректировки ставки", unitsBefore - unitsAfter, equalTo(GET_COST + ITEM_COST_ON_GET * 2));
    }

    @Test
    @Description("Получение нескольких корректировок ставок")
    public void getSeveralBidModifiersTest() {
        api.as(singleClient);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.bidModifiersSteps().bidModifiersGetById(bmID1, bmID2);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость получения двух корректировок ставок", unitsBefore - unitsAfter, equalTo(GET_COST + ITEM_COST_ON_GET * 2));
    }

    @Test
    @Description("Получение корректировки ставки по CampaignId и Type")
    public void getBidModifierByCampaignIDAndTypeTest() {
        api.as(singleClient);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.bidModifiersSteps().bidModifiersGet(new GetRequestMap()
                .withSelectionCriteria(new BidModifiersSelectionCriteriaMap()
                        .withCampaignIds(campaignID)
                        .withLevels(BidModifierLevelEnum.AD_GROUP)
                        .withTypes(BidModifierTypeEnum.DEMOGRAPHICS_ADJUSTMENT))
                .withAllFieldNames());
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        assertThat("стоимость получения одной корректировки ставки", unitsBefore - unitsAfter, equalTo(GET_COST + ITEM_COST_ON_GET * 2));
    }

    @Test
    @Description("Запрос несуществующей корректировки ставки")
    public void getNonExistingBidModifierTest() {
        api.as(singleClient);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.bidModifiersSteps().bidModifiersGetById(1234l);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость запроса несуществующей корректировки ставки", unitsBefore - unitsAfter, equalTo(GET_COST));
    }

    @Test
    @Description("Общая ошибка при получении корректировки ставки")
    public void ogeneralErrorOnGetBidModifierTest() {
        api.as(singleClient);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap()
                                .withLevels(BidModifierLevelEnum.AD_GROUP)
                                .withIds(bmID1)),
                new Api5Error(8000, Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                        "params",
                        StringUtils.capitalize(GetRequestMap.FIELD_NAMES)),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        StringUtils.capitalize(GetRequestMap.FIELD_NAMES)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость общей ошибки при получении корректировки ставки", unitsBefore - unitsAfter, equalTo(GENERAL_FAULT_COST));
    }

    @Test
    @Description("Недостаточное колличество баллов при получении корректировки ставки")
    public void notEnoughUnitsOnGetBidModifierTest() {
        api.as(singleClient);
        log.info("Обнулим колличество доступных пользователю баллов");
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(singleClient, unitsBefore);
        unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        assumeThat("доступные баллы пользователя", unitsBefore, equalTo(0));

        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new BidModifiersSelectionCriteriaMap()
                                .withLevels(BidModifierLevelEnum.AD_GROUP)
                                .withIds(bmID1))
                        .withAllFieldNames(),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        assertThat("доступные баллы пользователя после ошибки", unitsAfter, equalTo(0));
    }
}
