package ru.yandex.autotests.direct.api.bidmodifiers.set;

import java.util.List;

import com.yandex.direct.api.v5.bidmodifiers.BidModifierGetItem;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierSetMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.SerpLayoutAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.SetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 31.07.15
 */
@Aqua.Test
@Features(BidModifiersFeatures.SET)
@Tag(TagDictionary.TRUNK)
@Description("Проверка позитивных сценариев обновления корректировок ставок")
public class SetBidModifiersPositiveTest {
    private static final String client = BidModifiersLogins.SINGLE_CLIENT_SET;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long groupID;
    private static Long campaignID;

    @BeforeClass
    public static void createCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void prepareGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
    }

    @Test
    @Description("Обновление мобильной корректировки ставок")
    public void setMobileModifierTest() {
        Long bmID = api.userSteps.bidModifiersSteps().addBidModifierMobileToGroup(groupID);
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(
                        new BidModifierSetMap()
                                .defaultBidModifier()
                                .withId(bmID)),
                ExpectedResult.success(bmID));
        List<BidModifierGetItem> bms = api.userSteps.bidModifiersSteps().bidModifiersGetById(bmID);
        assumeThat("получили корректировку ставок", bms, hasSize(1));
        assumeThat("получили набор мобильных корректировок", bms.get(0).getMobileAdjustment(), notNullValue());

        assertThat("мобильная корректировка ставок изменена",
                bms.get(0).getMobileAdjustment().getBidModifier(),
                equalTo(BidModifierSetMap.DEFAULT_BID_MODIFIER));
    }

    @Test
    @Description("Обновление демографической корректировки ставок")
    public void setDemographicsModifierTest() {
        Long bmID = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID);
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(
                        new BidModifierSetMap()
                                .defaultBidModifier()
                                .withId(bmID)),
                ExpectedResult.success(bmID));
        List<BidModifierGetItem> bms = api.userSteps.bidModifiersSteps().bidModifiersGetById(bmID);
        assumeThat("получили корректировку ставок", bms, hasSize(1));
        assumeThat("получили набор корректировок по демографии", bms.get(0).getDemographicsAdjustment(),
                notNullValue());

        assertThat("корректировка ставок по демографии изменена",
                bms.get(0).getDemographicsAdjustment().getBidModifier(),
                equalTo(BidModifierSetMap.DEFAULT_BID_MODIFIER));
    }

    @Test
    @Description("Обновление корректировки ставок ретаргетинга")
    public void setRetargetingModifierTest() {
        Integer conditionId = api.userSteps.retargetingSteps().addRandomRetargetingCondition(client);
        Long bmID = api.userSteps.bidModifiersSteps().addBidModifierRetargetingToGroup(conditionId.longValue(),
                groupID);
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(
                        new BidModifierSetMap()
                                .defaultBidModifier()
                                .withId(bmID)),
                ExpectedResult.success(bmID));
        List<BidModifierGetItem> bms = api.userSteps.bidModifiersSteps().bidModifiersGetById(bmID);
        assumeThat("получили корректировку ставок", bms, hasSize(1));
        assumeThat("получили набор корректировок ретаргетинга", bms.get(0).getRetargetingAdjustment(), notNullValue());

        assertThat("корректировка ставок ретаргетинга изменена",
                bms.get(0).getRetargetingAdjustment().getBidModifier(),
                equalTo(BidModifierSetMap.DEFAULT_BID_MODIFIER));
    }

    @Test
    @Description("Обновление корректировки ставок по региону")
    public void setRegionalModifierTest() {
        Long bmID = api.userSteps.bidModifiersSteps().addBidModifierRegionalToCampaign(campaignID);
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(
                        new BidModifierSetMap()
                                .withBidModifier(BidModifierSetMap.DEFAULT_BID_MODIFIER / 2)
                                .withId(bmID)),
                ExpectedResult.success(bmID));
        List<BidModifierGetItem> bms = api.userSteps.bidModifiersSteps().bidModifiersGetById(bmID);
        assumeThat("получили корректировку ставок", bms, hasSize(1));
        assumeThat("получили набор корректировок по региону", bms.get(0).getRegionalAdjustment(), notNullValue());

        assertThat("корректировка ставок по региону изменена",
                bms.get(0).getRegionalAdjustment().getBidModifier(),
                equalTo(BidModifierSetMap.DEFAULT_BID_MODIFIER / 2));
    }

    @Test
    @Description("Обновление видео корректировки ставок на группу")
    public void setAdGroupVideoModifierTest() {
        Long bmID = api.userSteps.bidModifiersSteps().addBidModifierVideoToGroup(groupID);
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(
                        new BidModifierSetMap()
                                .withBidModifier(BidModifierSetMap.DEFAULT_BID_MODIFIER / 2)
                                .withId(bmID)),
                ExpectedResult.success(bmID));
        List<BidModifierGetItem> bms = api.userSteps.bidModifiersSteps().bidModifiersGetById(bmID);
        assumeThat("получили корректировку ставок", bms, hasSize(1));
        assumeThat("получили набор видео корректировок", bms.get(0).getVideoAdjustment(), notNullValue());

        assertThat("видео корректировка ставок изменена",
                bms.get(0).getVideoAdjustment().getBidModifier(),
                equalTo(BidModifierSetMap.DEFAULT_BID_MODIFIER / 2));
    }

    @Test
    @Description("Обновление видео корректировки ставок на кампании")
    public void setCampaignVideoModifierTest() {
        Long bmID = api.userSteps.bidModifiersSteps().addBidModifierVideoToCampaign(campaignID);
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(
                        new BidModifierSetMap()
                                .withBidModifier(BidModifierSetMap.DEFAULT_BID_MODIFIER / 2)
                                .withId(bmID)),
                ExpectedResult.success(bmID));
        List<BidModifierGetItem> bms = api.userSteps.bidModifiersSteps().bidModifiersGetById(bmID);
        assumeThat("получили корректировку ставок", bms, hasSize(1));
        assumeThat("получили набор видео корректировок", bms.get(0).getVideoAdjustment(), notNullValue());

        assertThat("видео корректировка ставок изменена",
                bms.get(0).getVideoAdjustment().getBidModifier(),
                equalTo(BidModifierSetMap.DEFAULT_BID_MODIFIER / 2));
    }

    @Test
    @Description("Обновление корректировки ставок на позицию")
    public void setSerpLayoutModifierTest() {
        Long bmID = api.userSteps.bidModifiersSteps().addBidModifierSerpLayoutToCampaign(campaignID);
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(
                        new BidModifierSetMap()
                                .withBidModifier(SerpLayoutAdjustmentMap.SERP_LAYOUT_ADJUSTMENT_MAX / 2)
                                .withId(bmID)),
                ExpectedResult.success(bmID));
        List<BidModifierGetItem> bms = api.userSteps.bidModifiersSteps().bidModifiersGetById(bmID);
        assumeThat("получили корректировку ставок", bms, hasSize(1));
        assumeThat("получили набор корректировок по региону", bms.get(0).getSerpLayoutAdjustment(), notNullValue());

        assertThat("корректировка ставок по региону изменена",
                bms.get(0).getSerpLayoutAdjustment().getBidModifier(),
                equalTo(SerpLayoutAdjustmentMap.SERP_LAYOUT_ADJUSTMENT_MAX / 2));
    }

    @Test
    @Description("Обновление корректировки ставок значением, которое уже установлено")
    public void sameModifierValueTest() {
        Long bmID = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToCampaign(campaignID.intValue());
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(
                        new BidModifierSetMap()
                                .withBidModifier(BidModifierAddMap.DEFAULT_BID_MODIFIER)
                                .withId(bmID)),
                ExpectedResult.success(bmID));
    }
}
