package ru.yandex.autotests.direct.api.bidmodifiers.set;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.enums.CampaignType;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierSetMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.DemographicsAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.MobileAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.VideoAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.RetargetingAdjustmentMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.SetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 08.08.15.
 */
@Aqua.Test
@Features(BidModifiersFeatures.SET)
@Description("Проверка допустимого диапазона корректировок ставок при модификации")
@RunWith(Parameterized.class)
public class SetBidModifiersRangeTest {
    private static final String client = BidModifiersLogins.SINGLE_CLIENT_SET;

    private static Long textCampaignID;
    private static Long dynamicCampaignID;
    private static Long mobileCampaignID;
    private Long groupID;
    private Long bmID;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public CampaignType type;

    @Parameterized.Parameter(2)
    public BidModifierAddMap bidModifier;

    @Parameterized.Parameter(3)
    public Integer newModifierValue;

    @Parameterized.Parameter(4)
    public ExpectedResult expectedResult;

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection<Object[]> data() {
        Integer conditionId = api.userSteps.retargetingSteps().addRandomRetargetingCondition(client);
        return Arrays.asList(new Object[][]{
                {"Мобильный - значение меньше минимального для текстовой кампании", CampaignType.TEXT, new BidModifierAddMap()
                        .withMobileAdjustment(new MobileAdjustmentMap().defaultMobileAdjustment()),
                        MobileAdjustmentMap.MOBILE_ADJUSTMENT_ABSOLUTE_MIN - 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER))},
                {"Мобильный - значение больше максимального для текстовой кампании", CampaignType.TEXT, new BidModifierAddMap()
                        .withMobileAdjustment(new MobileAdjustmentMap().defaultMobileAdjustment()),
                        MobileAdjustmentMap.MOBILE_ADJUSTMENT_MAX + 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.BID_MODIFIER_SHOULD_BE_NOT_MORE,
                                MobileAdjustmentMap.MOBILE_ADJUSTMENT_MAX))},
                {"Демографический - значение меньше минимального для текстовой кампании", CampaignType.TEXT, new BidModifierAddMap()
                        .withDemographicsAdjustment(new DemographicsAdjustmentMap().defaultDemographicsAdjustment()),
                        DemographicsAdjustmentMap.DEMOGRAPHICS_ADJUSTMENT_MIN - 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER))},
                {"Демографический - значение больше максимального для текстовой кампании", CampaignType.TEXT, new BidModifierAddMap()
                        .withDemographicsAdjustment(new DemographicsAdjustmentMap().defaultDemographicsAdjustment()),
                        DemographicsAdjustmentMap.DEMOGRAPHICS_ADJUSTMENT_MAX + 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.BID_MODIFIER_SHOULD_BE_NOT_MORE,
                                DemographicsAdjustmentMap.DEMOGRAPHICS_ADJUSTMENT_MAX))},
                {"Ретаргетинговый - значение меньше минимального для текстовой кампании", CampaignType.TEXT, new BidModifierAddMap()
                        .withRetargetingAdjustment(new RetargetingAdjustmentMap()
                                .defaultRetargetingAdjustment()
                                .withRetargetingConditionId(conditionId)),
                        RetargetingAdjustmentMap.RETARGETING_ADJUSTMENT_MIN - 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER))},
                {"Ретаргетинговый - значение больше максимального для текстовой кампании", CampaignType.TEXT, new BidModifierAddMap()
                        .withRetargetingAdjustment(new RetargetingAdjustmentMap()
                                .defaultRetargetingAdjustment()
                                .withRetargetingConditionId(conditionId)),
                        RetargetingAdjustmentMap.RETARGETING_ADJUSTMENT_MAX + 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.BID_MODIFIER_SHOULD_BE_NOT_MORE,
                                RetargetingAdjustmentMap.RETARGETING_ADJUSTMENT_MAX))},
                {"Видео - значение меньше минимального для текстовой кампании", CampaignType.TEXT, new BidModifierAddMap()
                        .withVideoAdjustment(new VideoAdjustmentMap().defaultVideoAdjustment()),
                        VideoAdjustmentMap.VIDEO_ADJUSTMENT_MIN - 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.BID_MODIFIER_SHOULD_BE_NOT_LESS,
                                VideoAdjustmentMap.VIDEO_ADJUSTMENT_MIN))},
                {"Видео - значение больше максимального для текстовой кампании", CampaignType.TEXT, new BidModifierAddMap()
                        .withVideoAdjustment(new VideoAdjustmentMap().defaultVideoAdjustment()),
                        VideoAdjustmentMap.VIDEO_ADJUSTMENT_MAX + 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.BID_MODIFIER_SHOULD_BE_NOT_MORE,
                                VideoAdjustmentMap.VIDEO_ADJUSTMENT_MAX))},
                {"Мобильный - значение меньше минимального для динамической кампании", CampaignType.DYNAMIC, new BidModifierAddMap()
                        .withMobileAdjustment(new MobileAdjustmentMap().defaultMobileAdjustment()),
                        MobileAdjustmentMap.MOBILE_ADJUSTMENT_ABSOLUTE_MIN - 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER))},
                {"Мобильный - значение больше максимального для динамической кампании", CampaignType.DYNAMIC, new BidModifierAddMap()
                        .withMobileAdjustment(new MobileAdjustmentMap().defaultMobileAdjustment()),
                        MobileAdjustmentMap.MOBILE_ADJUSTMENT_MAX + 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.BID_MODIFIER_SHOULD_BE_NOT_MORE,
                                MobileAdjustmentMap.MOBILE_ADJUSTMENT_MAX))},
                {"Демографический - значение меньше минимального для динамической кампании", CampaignType.DYNAMIC, new BidModifierAddMap()
                        .withDemographicsAdjustment(new DemographicsAdjustmentMap().defaultDemographicsAdjustment()),
                        DemographicsAdjustmentMap.DEMOGRAPHICS_ADJUSTMENT_MIN - 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER))},
                {"Демографический - значение больше максимального для динамической кампании", CampaignType.DYNAMIC, new BidModifierAddMap()
                        .withDemographicsAdjustment(new DemographicsAdjustmentMap().defaultDemographicsAdjustment()),
                        DemographicsAdjustmentMap.DEMOGRAPHICS_ADJUSTMENT_MAX + 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.BID_MODIFIER_SHOULD_BE_NOT_MORE,
                                DemographicsAdjustmentMap.DEMOGRAPHICS_ADJUSTMENT_MAX))},
                {"Ретаргетинговый - значение меньше минимального для динамической кампании", CampaignType.DYNAMIC, new BidModifierAddMap()
                        .withRetargetingAdjustment(new RetargetingAdjustmentMap()
                        .defaultRetargetingAdjustment()
                        .withRetargetingConditionId(conditionId)),
                        RetargetingAdjustmentMap.RETARGETING_ADJUSTMENT_MIN - 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER))},
                {"Ретаргетинговый - значение больше максимального для динамической кампании", CampaignType.DYNAMIC, new BidModifierAddMap()
                        .withRetargetingAdjustment(new RetargetingAdjustmentMap()
                        .defaultRetargetingAdjustment()
                        .withRetargetingConditionId(conditionId)),
                        RetargetingAdjustmentMap.RETARGETING_ADJUSTMENT_MAX + 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.BID_MODIFIER_SHOULD_BE_NOT_MORE,
                                RetargetingAdjustmentMap.RETARGETING_ADJUSTMENT_MAX))},
                {"Демографический - значение меньше минимального для мобильной кампании", CampaignType.MOBILE_CONTENT, new BidModifierAddMap()
                        .withDemographicsAdjustment(new DemographicsAdjustmentMap().defaultDemographicsAdjustment()),
                        DemographicsAdjustmentMap.DEMOGRAPHICS_ADJUSTMENT_MIN - 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER))},
                {"Демографический - значение больше максимального для мобильной кампании", CampaignType.MOBILE_CONTENT, new BidModifierAddMap()
                        .withDemographicsAdjustment(new DemographicsAdjustmentMap().defaultDemographicsAdjustment()),
                        DemographicsAdjustmentMap.DEMOGRAPHICS_ADJUSTMENT_MAX + 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.BID_MODIFIER_SHOULD_BE_NOT_MORE,
                                DemographicsAdjustmentMap.DEMOGRAPHICS_ADJUSTMENT_MAX))},
                {"Ретаргетинговый - значение меньше минимального для мобильной кампании", CampaignType.MOBILE_CONTENT, new BidModifierAddMap()
                        .withRetargetingAdjustment(new RetargetingAdjustmentMap()
                        .defaultRetargetingAdjustment()
                        .withRetargetingConditionId(conditionId)),
                        RetargetingAdjustmentMap.RETARGETING_ADJUSTMENT_MIN - 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.COEFFICIENT_VALUE_MUST_BE_A_POSITIVE_INTEGER))},
                {"Ретаргетинговый - значение больше максимального для мобильной кампании", CampaignType.MOBILE_CONTENT, new BidModifierAddMap()
                        .withRetargetingAdjustment(new RetargetingAdjustmentMap()
                        .defaultRetargetingAdjustment()
                        .withRetargetingConditionId(conditionId)),
                        RetargetingAdjustmentMap.RETARGETING_ADJUSTMENT_MAX + 1,
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.BID_MODIFIER_SHOULD_BE_NOT_MORE,
                                RetargetingAdjustmentMap.RETARGETING_ADJUSTMENT_MAX))}
        });
    }

    @BeforeClass
    public static void prepareCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        mobileCampaignID = api.as(client).userSteps.campaignSteps().addDefaultMobileAppCampaign(client);
        textCampaignID = api.as(client).userSteps.campaignSteps().addDefaultTextCampaign(client);
        dynamicCampaignID = api.as(client).userSteps.campaignSteps().addDefaultDynamicTextCampaign(client);
    }

    @Before
    public void prepareGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        if(type == CampaignType.TEXT){
            groupID = api.userSteps.adGroupsSteps().addDefaultGroup(textCampaignID);
        } else if(type == CampaignType.DYNAMIC){
            groupID = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(dynamicCampaignID);
        } else if (type == CampaignType.MOBILE_CONTENT){
            groupID = api.userSteps.adGroupsSteps().addDefaultMobileGroup(mobileCampaignID);
        }
        List<Long> bmIDs = api.userSteps.bidModifiersSteps().bidModifiersAdd(bidModifier.withAdGroupId(groupID));
        assumeThat("создана корректировка ставок", bmIDs, hasSize(1));
        bmID = bmIDs.get(0);
    }

    @Test
    public void bidModifierRangeTest() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(new BidModifierSetMap()
                        .withId(bmID)
                        .withBidModifier(newModifierValue)),
                expectedResult);
    }
}
