package ru.yandex.autotests.direct.api.bidmodifiers.set;

import org.apache.commons.lang3.StringUtils;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierSetMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.SetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;


/**
 * Created by chicos on 07.08.15
 */
@Aqua.Test
@Features(BidModifiersFeatures.SET)
@Description("Проверка некорректной структуры запроса Set")
public class SetBidModifiersStructureNegativeTest {

    private static final String CLIENT = BidModifiersLogins.SINGLE_CLIENT_SET;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    public static void prepareBidModifier() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(CLIENT);
        api.userSteps.bidModifiersSteps().addBidModifierMobileToCampaign(campaignID);
    }

    @Before
    public void clearUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
    }

    @Test
    public void noBidModifierElementInRequestTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.SET,
                new SetRequestMap(),
                JavaOrPerlApi5Error.java(
                        new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                                StringUtils.capitalize(SetRequestMap.BID_MODIFIERS)))
                        .perl(
                                new Api5Error(8000, Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                                        "params",
                                        StringUtils.capitalize(SetRequestMap.BID_MODIFIERS))));
    }

    @Test
    public void emptyBidModifiersPackTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(null),
                JavaOrPerlApi5Error.java(
                        new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_NOT_LESS_ELEMENTS,
                                StringUtils.capitalize(SetRequestMap.BID_MODIFIERS), 1))
                        .perl(
                                new Api5Error(8000, Api5ErrorDetails.FIELD_MUST_CONTAIN_NOT_LESS_ELEMENTS,
                                        StringUtils.capitalize(SetRequestMap.BID_MODIFIERS))));
    }

    @Test
    @Description("Делаем вызов метода Set, не передавая параметров")
    public void noParamsRequestTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.SET,
                null,
                JavaOrPerlApi5Error.java(new Api5Error(8000, Api5ErrorDetailsJava.INVALID_REQUEST_FORMAT))
                        .perl(new Api5Error(8000, Api5ErrorDetails.CANNOT_HAVE_NULL_VALUE, "params")));
    }

    @Test
    @Description("Максимальное колличество корректировок ставок в запросе")
    public void maxBidModifiersInOneRequestTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(getBidModifiersMapArray(123456)),
                new Api5Error(9300, Api5ErrorDetails.BID_MODIFIERS_SET_IN_REQUEST_LIMIT)
        );
    }

    private BidModifierSetMap[] getBidModifiersMapArray(long bmID) {
        BidModifierSetMap[] bidModifiers = new BidModifierSetMap[SetRequestMap.BID_MODIFIERS_IN_REQUEST_LIMIT + 1];
        for (int i = 0; i < SetRequestMap.BID_MODIFIERS_IN_REQUEST_LIMIT + 1; i++) {
            bidModifiers[i] = new BidModifierSetMap()
                    .defaultBidModifier()
                    .withId(bmID);
        }
        return bidModifiers;
    }
}
