package ru.yandex.autotests.direct.api.bidmodifiers.set;

import org.apache.commons.lang3.StringUtils;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierSetMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.SetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 07.08.15
 */
@Aqua.Test
@Features(BidModifiersFeatures.SET)
@Stories(ApiStories.UNITS)
@Tag(TagDictionary.TRUNK)
@Description("Списание баллов при модификации корректировок ставок")
public class SetBidModifiersUnitsTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());
    private static final String client = BidModifiersLogins.CLIENT_UNITS_SET;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private final int SET_COST = 2;
    private final int ITEM_COST_ON_SET = 0;
    private final int ITEM_COST_ON_SET_FAULT = 0;
    private final int GENERAL_FAULT_COST = 50;

    private static final Long INVALID_ID = 123456L;
    private static final Long INVALID_ID_2 = 123457L;

    private long groupID;
    private static Long campaignID;

    @BeforeClass
    public static void prepareCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
    }

    @Before
    public void prepareGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
    }

    @Test
    @Description("Модификация одиночной корректировки ставок")
    public void setSingleBidModifier() {
        Long bmId = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.bidModifiersSteps().bidModifiersSet(bmId, BidModifierSetMap.DEFAULT_BID_MODIFIER);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость модификации одной корректировки ставок", unitsBefore - unitsAfter, equalTo(SET_COST + ITEM_COST_ON_SET));
    }

    @Test
    @Description("Модификация нескольких корректировок ставок")
    public void setSeveralBidModifiersTest() {
        Integer conditionId = api.userSteps.retargetingSteps().addRandomRetargetingCondition(client);
        Long bmID1 = api.userSteps.bidModifiersSteps().addBidModifierRetargetingToGroup(conditionId.longValue(), groupID);
        Long bmID2 = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.bidModifiersSteps().bidModifiersSet(new SetRequestMap().withBidModifiers(
                new BidModifierSetMap()
                        .defaultBidModifier()
                        .withId(bmID1),
                new BidModifierSetMap()
                        .defaultBidModifier()
                        .withId(bmID2)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость удаления двух корректировок ставок", unitsBefore - unitsAfter, equalTo(SET_COST + ITEM_COST_ON_SET * 2));
    }

    @Test
    @Description("Модификация одного невалидной корректировки ставок")
    public void setInvalidBidModifierTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(
                        new BidModifierSetMap()
                                .defaultBidModifier()
                                .withId(INVALID_ID)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.BID_MODIFIER_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при модификации корректировки ставки", unitsBefore - unitsAfter, equalTo(SET_COST + ITEM_COST_ON_SET_FAULT));
    }

    @Test
    @Description("Модификации двух невалидных корректировок ставок")
    public void setFewInvalidBidModifiersTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(
                        new BidModifierSetMap()
                                .defaultBidModifier()
                                .withId(INVALID_ID),
                        new BidModifierSetMap()
                                .defaultBidModifier()
                                .withId(INVALID_ID_2)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.BID_MODIFIER_NOT_FOUND)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.BID_MODIFIER_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при модификации двух корректировок ставок", unitsBefore - unitsAfter, equalTo(SET_COST + ITEM_COST_ON_SET_FAULT * 2));
    }

    @Test
    @Description("Модификации одной корректной и одной невалидной корректировки ставок")
    public void setOneValidAndOneInvalidBidModifiersTest() {
        Long bmId = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(
                        new BidModifierSetMap()
                                .defaultBidModifier()
                                .withId(bmId),
                        new BidModifierSetMap()
                                .defaultBidModifier()
                                .withId(INVALID_ID)),
                ExpectedResult.success(bmId),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.BID_MODIFIER_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при модификации нескольких корректировок ставок",
                unitsBefore - unitsAfter, equalTo(SET_COST + ITEM_COST_ON_SET + ITEM_COST_ON_SET_FAULT));
    }

    @Test
    @Description("Невалидный запрос на модификацию корректировок ставок")
    public void setErrorOnSetBidModifierTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.SET,
                new SetRequestMap(),
                JavaOrPerlApi5Error.java(
                        new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                                StringUtils.capitalize(SetRequestMap.BID_MODIFIERS)))
                        .perl(
                                new Api5Error(8000, Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                                        "params",
                                        StringUtils.capitalize(SetRequestMap.BID_MODIFIERS))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость общей ошибки при модификации корректировки ставок", unitsBefore - unitsAfter, equalTo(GENERAL_FAULT_COST));
    }

    @Test
    @Description("Недостаточное колличество баллов при модификации корректировки ставок")
    public void notEnoughUnitsOnSetBidModifierTest() {
        Long bmId = api.userSteps.bidModifiersSteps().addBidModifierMobileToGroup(groupID);

        log.info("Обнулим колличество доступных пользователю баллов");
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(client, unitsBefore);
        unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assumeThat("доступные баллы пользователя", unitsBefore, equalTo(0));

        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.SET,
                new SetRequestMap().withBidModifiers(
                        new BidModifierSetMap()
                                .defaultBidModifier()
                                .withId(bmId)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("доступные баллы пользователя после ошибки", unitsAfter, equalTo(0));
    }
}
