package ru.yandex.autotests.direct.api.bidmodifiers.set;

import java.util.List;

import javax.annotation.Nullable;

import com.google.common.primitives.Shorts;
import com.yandex.direct.api.v5.bidmodifiers.OperatingSystemTypeEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.MobileAdjustmentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasProperty;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(BidModifiersFeatures.SET)
@Tag(TagDictionary.TRUNK)
@Description("Проверка состояния БД при изменении мобильных корректировок c таргетингом на ОС")
public class SetMobileBidModifierWithOsAndCheckDbTest {
    private static final String CLIENT = BidModifiersLogins.SINGLE_CLIENT_SET;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static DirectJooqDbSteps jooqDbSteps;
    private static Long cpmCampaignId;

    @BeforeClass
    public static void createCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT);
        cpmCampaignId = api.userSteps.campaignSteps().addDefaultCpmBannerCampaign();
    }

    @Before
    @Step("Подготовка данных для теста")
    public void prepareGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);

    }

    @Test
    public void setNewValueToMobileAdjustmentWithoutOsAndCheckRecordInHierarchicalMultipliersTable() {
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultCpmBannerKeywordsAdGroup(cpmCampaignId);
        Long bidModifierId = createMobileBidModifier(adGroupId, null);

        api.userSteps.bidModifiersSteps().bidModifiersSet(
                bidModifierId, MobileAdjustmentMap.MOBILE_ADJUSTMENT_ABSOLUTE_MIN);

        assertThat("Запись в БД имеет правильный коэффициент",
                jooqDbSteps.multipliersSteps().getHierarchicalMultipliersByPid(adGroupId),
                contains(hasProperty("multiplierPct",
                        equalTo(Shorts.checkedCast(MobileAdjustmentMap.MOBILE_ADJUSTMENT_ABSOLUTE_MIN))))
        );
    }

    @Test
    public void setNewValueToMobileAdjustmentWithOsAndCheckRecordInMobileMultiplierValuesTable() {
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultCpmBannerKeywordsAdGroup(cpmCampaignId);
        Long bidModifierId = createMobileBidModifier(adGroupId, OperatingSystemTypeEnum.IOS);

        api.userSteps.bidModifiersSteps().bidModifiersSet(
                bidModifierId, MobileAdjustmentMap.MOBILE_ADJUSTMENT_ABSOLUTE_MIN);

        assertThat("Запись в БД имеет правильный коэффициент",
                jooqDbSteps.multipliersSteps().getMobileMultiplierValuesByAdGroupId(adGroupId),
                contains(hasProperty("multiplierPct",
                        equalTo(Shorts.checkedCast(MobileAdjustmentMap.MOBILE_ADJUSTMENT_ABSOLUTE_MIN))))
        );
    }

    private Long createMobileBidModifier(Long adGroupId, @Nullable OperatingSystemTypeEnum osType) {
        List<Long> ids = api.userSteps.bidModifiersSteps().bidModifiersAdd(
                new BidModifierAddMap()
                        .withAdGroupId(adGroupId)
                        .withMobileAdjustment(new MobileAdjustmentMap()
                                .withOperatingSystemType(osType)
                                .withBidModifier(BidModifierAddMap.DEFAULT_BID_MODIFIER))
        );
        assumeThat("Создана одна корректировка", ids, contains(notNullValue()));
        return ids.get(0);
    }
}
