package ru.yandex.autotests.direct.api.bidmodifiers.toggle;

import com.yandex.direct.api.v5.bidmodifiers.BidModifierToggleTypeEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.CampaignFakeInfo;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.GroupFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierToggleMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.ToggleExpectedResult;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.ToggleRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 11.08.15.
 */
@Aqua.Test
@Features(BidModifiersFeatures.TOGGLE)
@Tag(TagDictionary.TRUNK)
@Description("Проверка у группы времени последнего изменения и статуса синхронизации с БК при отключении набора корректировок")
public class ToggleBidModifiersBsSyncedGroupTest {
    private static LogSteps log = LogSteps.getLogger(ToggleBidModifiersBsSyncedGroupTest.class);
    private static final String client = BidModifiersLogins.SINGLE_CLIENT_TOGGLE;

    private static DateTime lastChange = DateTime.now().minusDays(5);
    private static final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");

    private static Long campaignID;
    private static Long groupID;
    private static Long bmID;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepareGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        Long bannerID = api.userSteps.adsSteps().addDefaultTextAd(groupID);
        bmID = api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID);

        log.info("Промодерируем фейково");
        api.userSteps.campaignFakeSteps().makeCampaignModerated(campaignID);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(groupID);
        api.userSteps.bannersFakeSteps().makeBannersModerated(bannerID);

        log.info("Выставим статусы синхронизации с БК фейково");
        api.userSteps.campaignFakeSteps().setBSSynced(campaignID, true);
        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(groupID, Status.YES);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(bannerID, Status.YES);

        log.info("Сдвинем время последнего изменения на несколько дней");
        api.userSteps.campaignFakeSteps().setLastChange(campaignID, lastChange.toString());
        api.userSteps.groupFakeSteps().setLastChange(groupID, lastChange.toString());
        api.userSteps.bannersFakeSteps().setLastChange(bannerID, lastChange.toString());

        log.info("Отключим демографический набор корректировок группы");
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.TOGGLE,
                new ToggleRequestMap().withBidModifierToggleItems(new BidModifierToggleMap()
                        .withType(BidModifierToggleTypeEnum.DEMOGRAPHICS_ADJUSTMENT)
                        .withEnabled(YesNoEnum.NO)
                        .withAdGroupId(groupID)),
                ToggleExpectedResult.successGroup(groupID));
    }

    @Test
    public void campaignStatusBsSyncedTest() {
        CampaignFakeInfo campaignInfo = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(campaignID);

        assertThat("статус синхронизации с БК у кампании после отключения набора корректировок",
                campaignInfo.getStatusBsSynced(), equalTo(Status.YES));
    }

    @Test
    public void groupStatusBsSyncedTest() {
        GroupFakeInfo groupInfo = api.userSteps.groupFakeSteps().getGroupParams(groupID);

        assertThat("статус синхронизации с БК у группы после отключения набора корректировок",
                groupInfo.getStatusBsSynced(), equalTo(Status.NO));
    }

    @Test
    public void campaignLastChangeTest() {
        CampaignFakeInfo campaignInfo = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(campaignID);
        assumeThat("время последнего изменения кампании", campaignInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(campaignInfo.getLastChange());

        assertThat("время последнего изменения кампании после отключения набора корректировок", date, equalTo(lastChange.toLocalDate()));
    }

    @Test
    public void groupLastChangeTest() {
        GroupFakeInfo groupInfo = api.userSteps.groupFakeSteps().getGroupParams(groupID);
        assumeThat("время последнего изменения группы", groupInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(groupInfo.getLastChange());

        assertThat("время последнего изменения группы после отключения набора корректировок", date, equalTo(lastChange.toLocalDate()));
    }
}
