package ru.yandex.autotests.direct.api.bidmodifiers.toggle;

import com.yandex.direct.api.v5.bidmodifiers.BidModifierToggleTypeEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import org.apache.commons.lang3.StringUtils;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierToggleMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.ToggleExpectedResult;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.ToggleRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;


/**
 * Created by chicos on 11.08.15
 */
@Aqua.Test
@Features(BidModifiersFeatures.TOGGLE)
@Description("Проверка негативных сценариев отключения набора корректировок ставок")
public class ToggleBidModifiersStructureNegativeTest {
    private static final String client = BidModifiersLogins.SINGLE_CLIENT_TOGGLE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long groupID;
    private static Long campaignID;

    @BeforeClass
    public static void createCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void prepareGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);

    }

    @Test
    public void noBidModifiersToggleItemInRequestTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.TOGGLE,
                new ToggleRequestMap(),
                JavaOrPerlApi5Error.java(
                        new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                                StringUtils.capitalize(ToggleRequestMap.BID_MODIFIER_TOGGLE_ITEMS)))
                        .perl(
                                new Api5Error(8000,
                                        Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                                        "params",
                                        StringUtils.capitalize(ToggleRequestMap.BID_MODIFIER_TOGGLE_ITEMS))));
    }

    @Test
    public void emptyBidModifierToggleItemsPackTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.TOGGLE,
                new ToggleRequestMap().withBidModifierToggleItems(null),
                JavaOrPerlApi5Error.java(
                        new Api5Error(8000,
                                Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_NOT_LESS_ELEMENTS,
                                StringUtils.capitalize(ToggleRequestMap.BID_MODIFIER_TOGGLE_ITEMS), 1))
                        .perl(
                                new Api5Error(8000,
                                        Api5ErrorDetails.FIELD_MUST_CONTAIN_NOT_LESS_ELEMENTS,
                                        StringUtils.capitalize(ToggleRequestMap.BID_MODIFIER_TOGGLE_ITEMS))));
    }

    @Test
    @Description("Попытка отключения набора корректировок без указания типа")
    public void noTypeFieldTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.TOGGLE,
                new ToggleRequestMap().withBidModifierToggleItems(
                        new BidModifierToggleMap()
                                .withEnabled(YesNoEnum.NO)
                                .withAdGroupId(groupID)),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.IN_ELEMENT_OF_ARRAY_MISSING_REQUIRED_FIELD,
                        StringUtils.capitalize(ToggleRequestMap.BID_MODIFIER_TOGGLE_ITEMS),
                        StringUtils.capitalize(BidModifierToggleMap.TYPE)));
    }


    @Test
    @Description("Попытка отключения набора корректировок без указания флага активации")
    public void noEnabledFieldTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.TOGGLE,
                new ToggleRequestMap().withBidModifierToggleItems(
                        new BidModifierToggleMap()
                                .withType(BidModifierToggleTypeEnum.DEMOGRAPHICS_ADJUSTMENT)
                                .withAdGroupId(groupID)),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.IN_ELEMENT_OF_ARRAY_MISSING_REQUIRED_FIELD,
                        StringUtils.capitalize(ToggleRequestMap.BID_MODIFIER_TOGGLE_ITEMS),
                        StringUtils.capitalize(BidModifierToggleMap.ENABLED)));
    }

    @Test
    @Description("Попытка отключения набора корректировок без идентификатора кампании/группы")
    public void noCampaignIdOrGroupIdTest() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.TOGGLE,
                new ToggleRequestMap().withBidModifierToggleItems(
                        new BidModifierToggleMap()
                                .withType(BidModifierToggleTypeEnum.DEMOGRAPHICS_ADJUSTMENT)
                                .withEnabled(YesNoEnum.NO)),
                ToggleExpectedResult.errors(new Notification(5008,
                        Api5ErrorDetails.ONE_OF_FIELDS_SHOULD_BE_SPECIFIED_BM,
                        BidModifierToggleMap.getBidModifiersMandatoryIdNames())));
    }

    @Test
    @Description("Попытка отключения набора корректировок с указанием идентификаторов кампании и группы в одном объекте")
    public void campaignIdAndGroupIdTest() {
        api.userSteps.bidModifiersSteps().shouldGetResultOn(
                Action.TOGGLE,
                new ToggleRequestMap().withBidModifierToggleItems(
                        new BidModifierToggleMap()
                                .withType(BidModifierToggleTypeEnum.DEMOGRAPHICS_ADJUSTMENT)
                                .withEnabled(YesNoEnum.NO)
                                .withAdGroupId(groupID)
                                .withCampaignId(campaignID)
                ),
                ToggleExpectedResult.errors(new Notification(5009,
                        Api5ErrorDetails.ONLY_ONE_OF_FIELDS_SHOULD_BE_SPECIFIED_BM,
                        BidModifierToggleMap.getBidModifiersMandatoryIdNames())));
    }

    @Test
    @Description("Количество элементов BidModifierToggleItem в запросе превышает максимальное")
    public void overlimitBidModifierToggleItemsInOneRequestTest() {
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.TOGGLE,
                new ToggleRequestMap().withBidModifierToggleItems(getBidModifiersMapArray(123456L)),
                new Api5Error(9300, Api5ErrorDetails.BID_MODIFIERS_TOGGLE_IN_REQUEST_LIMIT)
        );
    }

    private BidModifierToggleMap[] getBidModifiersMapArray(long campaignID) {
        BidModifierToggleMap[] bidModifiers =
                new BidModifierToggleMap[ToggleRequestMap.BID_MODIFIERS_IN_REQUEST_LIMIT + 1];
        for (int i = 0; i < ToggleRequestMap.BID_MODIFIERS_IN_REQUEST_LIMIT + 1; i++) {
            bidModifiers[i] = new BidModifierToggleMap()
                    .defaultBidModifier()
                    .withCampaignId(campaignID);
        }
        return bidModifiers;
    }
}
